<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of DotClear.
# Copyright (c) 2005 Olivier Meunier. All rights
# reserved.
#
# DotClear is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# DotClear is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with DotClear; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****

class dcXmlRpc extends IXR_IntrospectionServer
{
	public $core;
	private $blog_id;
	private $debug = false;
	private $debug_file = '/tmp/dotclear-xmlrpc.log';
	private $trace_args = true;
	private $trace_response = true;
	
	public function __construct(&$core,$blog_id)
	{
		$this->IXR_IntrospectionServer();
		
		$this->core =& $core;
		$this->blog_id = $blog_id;
		
		# Blogger methods
		$this->addCallback('blogger.newPost','this:blogger_newPost',
			array('string','string','string','string','string','string','integer'),
			'New post');
		
		$this->addCallback('blogger.editPost','this:blogger_editPost',
			array('boolean','string','string','string','string','string','integer'),
			'Edit a post');
		
		$this->addCallback('blogger.getPost','this:blogger_getPost',
			array('struct','string','integer','string','string'),
			'Return a posts by ID');
		
		$this->addCallback('blogger.deletePost','this:blogger_deletePost',
			array('string','string','string','string','string','integer'),
			'Delete a post');
		
		$this->addCallback('blogger.getRecentPosts','this:blogger_getRecentPosts',
			array('array','string','string','string','string','integer'),
			'Return a list of recent posts');
		
		$this->addCallback('blogger.getUsersBlogs','this:blogger_getUserBlogs',
			array('struct','string','string','string'),
			"Return user's blog");
		
		$this->addCallback('blogger.getUserInfo','this:blogger_getUserInfo',
			array('struct','string','string','string'),
			'Return User Info');
		
		# Metaweblog methods
		$this->addCallback('metaWeblog.newPost','this:mw_newPost',
			array('string','string','string','string','struct','boolean'),
			'Creates a new post, and optionnaly publishes it.');
		
		$this->addCallback('metaWeblog.editPost','this:mw_editPost',
			array('boolean','string','string','string','struct','boolean'),
			'Updates information about an existing entry');
		
		$this->addCallback('metaWeblog.getPost','this:mw_getPost',
			array('struct','string','string','string'),
			'Returns information about a specific post');
		
		$this->addCallback('metaWeblog.getRecentPosts','this:mw_getRecentPosts',
			array('array','string','string','string','integer'),
			'List of most recent posts in the system');
		
		$this->addCallback('metaWeblog.newMediaObject','this:mw_newMediaObject',
			array('string','string','string','string','struct'),
			'Download a file on the web server');
		
		# MovableType methods
		$this->addCallback('mt.getRecentPostTitles','this:mt_getRecentPostTitles',
			array('array','string','string','string','integer'),
			'List of most recent posts in the system');
		
		$this->addCallback('mt.getCategoryList','this:mt_getCategoryList',
			array('array','string','string','string'),
			'List of all categories defined in the weblog');
		
		$this->addCallback('mt.getPostCategories','this:mt_getPostCategories',
			array('array','string','string','string'),
			'List of all categories to which the post is assigned');
		
		$this->addCallback('mt.setPostCategories','this:mt_setPostCategories',
			array('boolean','string','string','string','array'),
			'Sets the categories for a post');
		
		$this->addCallback('mt.publishPost','this:mt_publishPost',
			array('boolean','string','string','string'),
			'Retrieve pings list for a post');
		
		$this->addCallback('mt.supportedMethods','this:listMethods',
			array(),'Retrieve information about the XML-RPC methods supported by the server.');
		
		$this->addCallback('mt.supportedTextFilters','this:mt_supportedTextFilters',
			array(),'Retrieve information about supported text filters.');
	}
	
	public function serve($data=false,$encoding='UTF-8')
	{
		parent::serve(false,$encoding);
	}
	
	public function call($methodname,$args)
	{
		if (!$this->debug)
		{
			return parent::call($methodname,$args);
		}
		else
		{
			$res = parent::call($methodname,$args);
			if (($fp = @fopen($this->debug_file,'a')) !== false)
			{
				fwrite($fp,'['.date('r').']'.' '.$methodname);
				
				if ($this->trace_args) {
					fwrite($fp,"\n- args ---\n".var_export($args,1));
				}
				
				if ($this->trace_response) {
					fwrite($fp,"\n- response ---\n".var_export($res,1));
				}
				fwrite($fp,"\n");
				fclose($fp);
			}
			
			return $res;
		}
	}
	
	
	/* Internal methods
	--------------------------------------------------- */
	private function setUser($user_id,$pwd)
	{
		if ($this->core->auth->checkUser($user_id,$pwd) !== true) {
			throw new Exception('Login error');
		}
		
		$this->core->getUserBlogs();
		return true;
	}
	
	private function setBlog()
	{
		if (!$this->blog_id) {
			throw new Exception('No blog ID given.');
		}
		
		$this->core->setBlog($this->blog_id);
		
		if (!$this->core->blog->id) {
			$this->core->blog = null;
			throw new Exception('Blog does not exist.');
		}
		
		if (!$this->core->blog->settings->enable_xmlrpc ||
		!$this->core->auth->check('usage,contentadmin',$this->core->blog->id)) {
			$this->core->blog = null;
			throw new Exception('Not enough permissions on this blog.');
		}
		
		$this->core->plugins->loadModules(DC_PLUGINS_ROOT,'xmlrpc',$this->core->auth->getInfo('user_lang'));
		return true;
	}
	
	private function getPostRS($post_id,$user,$pwd)
	{
		$this->setUser($user,$pwd);
		$this->setBlog();
		$rs = $this->core->blog->getPosts(array('post_id' => (integer) $post_id));
		
		if ($rs->isEmpty()) {
			throw new Exception('This entry does not exist');
		}
		
		return $rs;
	}
	
	/* Generic methods
	--------------------------------------------------- */
	private function newPost($blog_id,$user,$pwd,$content,$struct=array(),$publish=true)
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog();
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$title = !empty($struct['title']) ? $struct['title'] : '';
		$excerpt = !empty($struct['mt_excerpt']) ? $struct['mt_excerpt'] : '';
		$description = !empty($struct['description']) ? $struct['description'] : null;
		$dateCreated = !empty($struct['dateCreated']) ? $struct['dateCreated'] : null;
		$open_comment = isset($struct['mt_allow_comments']) ? $struct['mt_allow_comments'] : 1;
		$open_tb = isset($struct['mt_allow_pings']) ? $struct['mt_allow_pings'] : 1;
		
		if ($description !== null) {
			$content = $description;
		}
		
		if (!$title) {
			$title = text::cutString(html::clean($content),25).'...';
		}
		
		$excerpt_xhtml = $this->core->callFormater('xhtml',$excerpt);
		$content_xhtml = $this->core->callFormater('xhtml',$content);
		
		if (empty($content)) {
			return new IXR_Error(804,'Cannot create an empty entry');
		}
		
		$cur = $this->core->con->openCursor($this->core->prefix.'post');
		
		$cur->user_id = $this->core->auth->userID();
		$cur->post_lang = $this->core->auth->getInfo('user_lang');
		$cur->post_title = trim($title);
		$cur->post_content = $content;
		$cur->post_excerpt = $excerpt;
		$cur->post_content_xhtml = $content_xhtml;
		$cur->post_excerpt_xhtml = $excerpt_xhtml;
		$cur->post_open_comment = (integer) ($open_comment == 1);
		$cur->post_open_tb = (integer) ($open_tb == 1);
		$cur->post_status = (integer) $publish;
		$cur->post_format = 'xhtml';
		
		if ($dateCreated && strtotime($dateCreated)) {
			$cur->post_dt = date('Y-m-d H:i:00',strtotime($dateCreated));
		}
		
		try
		{
			# --BEHAVIOR-- xmlrpcBeforeNewPost
			$this->core->callBehavior('xmlrpcBeforeNewPost',$this,$cur,$content,$struct,$publish);
			
			$post_id = $this->core->blog->addPost($cur);
			
			# --BEHAVIOR-- xmlrpcAfterNewPost
			$this->core->callBehavior('xmlrpcAfterNewPost',$this,$post_id,$cur,$content,$struct,$publish);
		}
		catch (Exception $e)
		{
			return new IXR_Error(805,$e->getMessage());
		}
		
		return $post_id;
	}
	
	private function editPost($post_id,$user,$pwd,$content,$struct=array(),$publish=true)
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$title = (!empty($struct['title'])) ? $struct['title'] : '';
		$excerpt = (!empty($struct['mt_excerpt'])) ? $struct['mt_excerpt'] : '';
		$description = (!empty($struct['description'])) ? $struct['description'] : null;
		$dateCreated = !empty($struct['dateCreated']) ? $struct['dateCreated'] : null;
		$open_comment = (isset($struct['mt_allow_comments'])) ? $struct['mt_allow_comments'] : 1;
		$open_tb = (isset($struct['mt_allow_pings'])) ? $struct['mt_allow_pings'] : 1;
		
		if ($description !== null) {
			$content = $description;
		}
		
		if (!$title) {
			$title = text::cutString(html::clean($content),25).'...';
		}
		
		$excerpt_xhtml = $this->core->callFormater('xhtml',$excerpt);
		$content_xhtml = $this->core->callFormater('xhtml',$content);
		
		if (empty($content)) {
			return new IXR_Error(804,'Cannot create an empty entry');
		}
		
		$cur = $this->core->con->openCursor($this->core->prefix.'post');
		
		$cur->post_title = trim($title);
		$cur->post_content = $content;
		$cur->post_excerpt = $excerpt;
		$cur->post_content_xhtml = $content_xhtml;
		$cur->post_excerpt_xhtml = $excerpt_xhtml;
		$cur->post_open_comment = (integer) ($open_comment == 1);
		$cur->post_open_tb = (integer) ($open_tb == 1);
		$cur->post_status = (integer) $publish;
		$cur->post_format = 'xhtml';
		$cur->post_url = $post->post_url;
		
		if ($dateCreated && strtotime($dateCreated)) {
			$cur->post_dt = date('Y-m-d H:i:00',strtotime($dateCreated));
		} else {
			$cur->post_dt = $post->post_dt;
		}
		
		try
		{
			# --BEHAVIOR-- xmlrpcBeforeEditPost
			$this->core->callBehavior('xmlrpcBeforeEditPost',$this,$post_id,$cur,$content,$struct,$publish);
			
			$this->core->blog->updPost($post_id,$cur);
			
			# --BEHAVIOR-- xmlrpcAfterEditPost
			$this->core->callBehavior('xmlrpcAfterEditPost',$this,$post_id,$cur,$content,$struct,$publish);
		}
		catch (Exception $e)
		{
			return new IXR_Error(805,$e->getMessage());
		}
		
		return true;
	}
	
	private function getPost($post_id,$user,$pwd,$type='mw')
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$res = array();
		
		$res['dateCreated'] = new IXR_Date($post->getTS());
		$res['userid'] = $post->user_id;
		$res['postid'] = $post->post_id;
		
		if ($type == 'blogger') {
			$res['content'] = $post->post_content_xhtml;
		}
		
		if ($type == 'mt' || $type == 'mw') {
			$res['title'] = $post->post_title;
		}
		
		if ($type == 'mw') {
			$res['description'] = $post->post_content_xhtml;
			$res['link'] = $res['permaLink'] = $post->getURL();
			$res['mt_excerpt'] = $post->post_excerpt_xhtml;
			$res['mt_text_more'] = '';
			$res['mt_allow_comments'] = (integer) $post->post_open_comment;
			$res['mt_allow_pings'] = (integer) $post->post_open_tb;
			$res['mt_convert_breaks'] = '';
			$res['mt_keywords'] = '';
		}
		
		# --BEHAVIOR-- xmlrpcGetPostInfo
		$this->core->callBehavior('xmlrpcGetPostInfo',$this,$type,array(&$res));
		
		return $res;
	}
	
	private function deletePost($post_id,$user,$pwd)
	{
		$post_id = (integer) $post_id;
		
		try {
			$this->getPostRS($post_id,$user,$pwd);
			$this->core->blog->delPost($post_id);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return true;
	}
	
	private function getRecentPosts($blog_id,$user,$pwd,$nb_post,$type='mw')
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog();
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$nb_post = (integer) $nb_post;
		
		if ($nb_post > 50) {
			return new IXR_Error(805,'Cannot retrieve more than 50 entries');
		}
		
		$params = array();
		$params['limit'] = $nb_post;
		
		try {
			$posts = $this->core->blog->getPosts($params);
		} catch (Exception $e) {
			return new IXR_Error(805,$e->getMessage());
		}
		
		$res = array();
		while ($posts->fetch())
		{
			$tres = array();
			
			$tres['dateCreated'] = new IXR_Date($posts->getTS());
			$tres['userid'] = $posts->user_id;
			$tres['postid'] = $posts->post_id;
			
			if ($type == 'blogger') {
				$tres['content'] = $posts->post_content_xhtml;
			}
			
			if ($type == 'mt' || $type == 'mw') {
				$tres['title'] = $posts->post_title;
			}
			
			if ($type == 'mw') {
				$tres['description'] = $posts->post_content_xhtml;
				$tres['link'] = $tres['permaLink'] = $posts->getURL();
				$tres['mt_excerpt'] = $posts->post_excerpt_xhtml;
				$tres['mt_text_more'] = '';
				$tres['mt_allow_comments'] = (integer) $posts->post_open_comment;
				$tres['mt_allow_pings'] = (integer) $posts->post_open_tb;
				$tres['mt_convert_breaks'] = '';
				$tres['mt_keywords'] = '';
			}
			
			# --BEHAVIOR-- xmlrpcGetPostInfo
			$this->core->callBehavior('xmlrpcGetPostInfo',$this,$type,array(&$tres));
			
			$res[] = $tres;
		}
		
		return $res;
	}
	
	private function getUsersBlogs($user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog();
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return array(array(
			'url' => $this->core->blog->url,
			'blogid' => '1',
			'blogName' => $this->core->blog->name
		));
	}
	
	private function getUserInfo($user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return array(
			'userid' => $this->core->auth->userID(),
			'firstname' => $this->core->auth->getInfo('user_firstname'),
			'lastname' => $this->core->auth->getInfo('user_name'),
			'nickname' => $this->core->auth->getInfo('user_displayname'),
			'email' => $this->core->auth->getInfo('user_email'),
			'url' => $this->core->auth->getInfo('user_url')
		);
	}
	
	private function getCategoryList($blog_id,$user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog();
			$rs = $this->core->blog->getCategories();
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$res = array();
		while ($rs->fetch()) {
			$res[] = array(
				'categoryId' => (string) $rs->cat_id,
				'categoryName' => $rs->cat_title
			);
		}
		
		return $res;
	}
	
	private function getPostCategories($post_id,$user,$pwd)
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return array(
			array(
			'categoryName' => $post->cat_title,
			'categoryId' => (string) $post->cat_id,
			'isPrimary' => true
			)
		);
	}
	
	private function setPostCategories($post_id,$user,$pwd,$categories)
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$cat_id = (!empty($categories[0]['categoryId'])) ? (integer) $categories[0]['categoryId'] : NULL;
		
		foreach($categories as $v)
		{
			if (isset($v['isPrimary']) && $v['isPrimary']) {
				$cat_id = $v['categoryId'];
				break;
			}
		}
		
		# w.bloggar sends -1 for no category.
		if ($cat_id == -1) {
			$cat_id = null;
		}
		
		try {
			$this->core->blog->updPostCategory($post_id,(integer) $cat_id);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return true;
	}
	
	private function publishPost($post_id,$user,$pwd)
	{
		$post_id = (integer) $post_id;
		
		try
		{
			$this->getPostRS($post_id,$user,$pwd);
			
			# --BEHAVIOR-- xmlrpcBeforePublishPost
			$this->core->callBehavior('xmlrpcBeforePublishPost',$this,$post_id);
			
			$this->core->blog->updPostStatus($post_id,1);
			
			# --BEHAVIOR-- xmlrpcAfterPublishPost
			$this->core->callBehavior('xmlrpcAfterPublishPost',$this,$post_id);
		}
		catch (Exception $e)
		{
			return new IXR_Error(801,$e->getMessage());
		}
		
		return true;
	}
	
	private function newMediaObject($blog_id,$user,$pwd,$file)
	{
		if (empty($file['name'])) {
			return new IXR_Error(801,'No file name');
		}
		
		if (empty($file['bits'])) {
			return new IXR_Error(801,'No file content');
		}
		
		$file_name = $file['name'];
		$file_bits = base64_decode($file['bits']);
		
		try
		{
			$this->setUser($user,$pwd);
			$this->setBlog();
			
			$media = new dcMedia($this->core);
			$media_id = $media->uploadBits($file_name,$file_bits);
			
			$f = $media->getFile($media_id);
			return array('url' => $f->file_url);
		}
		catch (Exception $e)
		{
			return new IXR_Error(801,$e->getMessage());
		}
	}
	
	/* Blogger methods
	--------------------------------------------------- */
	public function blogger_newPost($args)
	{
		return $this->newPost($args[1],$args[2],$args[3],$args[4],array(),$args[5]);
	}
	
	public function blogger_editPost($args)
	{
		return $this->editPost($args[1],$args[2],$args[3],$args[4],array(),$args[5]);
	}
	
	public function blogger_getPost($args)
	{
		return $this->getPost($args[1],$args[2],$args[3],'blogger');
	}
	
	public function blogger_deletePost($args)
	{
		return $this->deletePost($args[1],$args[2],$args[3]);
	}
	
	public function blogger_getRecentPosts($args)
	{
		return $this->getRecentPosts($args[1],$args[2],$args[3],$args[4],'blogger');
	}
	
	public function blogger_getUserBlogs($args)
	{
		return $this->getUsersBlogs($args[1],$args[2]);
	}
	
	public function blogger_getUserInfo($args)
	{
		return $this->getUserInfo($args[1],$args[2]);
	}
	
	
	/* Metaweblog methods
	------------------------------------------------------- */
	public function mw_newPost($args)
	{
		return $this->newPost($args[0],$args[1],$args[2],'',$args[3],$args[4]);
	}
	
	public function mw_editPost($args)
	{
		return $this->editPost($args[0],$args[1],$args[2],'',$args[3],$args[4]);
	}
	
	public function mw_getPost($args)
	{
		return $this->getPost($args[0],$args[1],$args[2],'mw');
	}
	
	public function mw_getRecentPosts($args)
	{
		return $this->getRecentPosts($args[0],$args[1],$args[2],$args[3],'mw');
	}
	
	public function mw_newMediaObject($args)
	{
		return $this->newMediaObject($args[0],$args[1],$args[2],$args[3]);
	}
	
	/* MovableType methods
	--------------------------------------------------- */
	public function mt_getRecentPostTitles($args)
	{
		return $this->getRecentPosts($args[0],$args[1],$args[2],$args[3],'mt');
	}
	
	public function mt_getCategoryList($args)
	{
		return $this->getCategoryList($args[0],$args[1],$args[2]);
	}
	
	public function mt_getPostCategories($args)
	{
		return $this->getPostCategories($args[0],$args[1],$args[2]);
	}
	
	public function mt_setPostCategories($args)
	{
		return $this->setPostCategories($args[0],$args[1],$args[2],$args[3]);
	}
	
	public function mt_publishPost($args)
	{
		return $this->publishPost($args[0],$args[1],$args[2]);
	}
	
	public function mt_supportedTextFilters($args)
	{
		return array();
	}
}
?>