<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of DotClear.
# Copyright (c) 2005 Olivier Meunier and contributors. All rights
# reserved.
#
# DotClear is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# DotClear is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with DotClear; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****

/**
@class dcCore

=== Introduction ===

True to its name dcCore is the core of !DotClear. It handles everything related
to blogs, database connection, plugins...

@param connection	con			connection object
@param string		prefix		tables prefix in database
@param dcBlog		blog			dcBlog object
@param dcError		error		dcError object
@param dcAuth		auth			dcAuth object
@param dcSession	session		dcSession object
@param urlHandler	url			urlHandler object
@param wiki2xhtml	wiki2xhtml	wiki2xhtml object
@param dcModules	plugins		dcModules object
@param array		blogs		Available blogs list
*/
class dcCore
{
	public $con;
	public $prefix;
	public $blog;
	public $error;
	public $auth;
	public $session;
	public $url;
	public $wiki2xhtml;
	public $plugins;
	public $media;
	
	private $formaters = array();
	private $behaviors = array();
	
	public $blogs;
	
	public $rest;
	
	/** @doc
	=== Common methods === */
	
	/**
	@function __construct
	
	@param	string	host			Database hostname
	@param	string	db			Database name
	@param	string	user			Database username
	@param	string	password		Database password
	@param	string	prefix		DotClear tables prefix
	*/
	public function __construct($driver, $host, $db, $user, $password, $prefix)
	{
		$this->con = dbLayer::init($driver,$host,$db,$user,$password);
		
		$this->prefix = $prefix;
		
		$this->error = new dcError();
		$this->auth = new dcAuth($this);
		$this->session = new dcSession($this,DC_SESSION_NAME);
		$this->url = new urlHandler($this);
		
		$this->plugins = new dcModules($this);
		
		$this->rest = new dcRestServer($this);
		
		$this->addFormater('xhtml', create_function('$s','return $s;'));
		$this->addFormater('wiki', array($this,'wikiTransform'));
	}
	
	public function __toString()
	{
		return 'DotClear core object';
	}
	
	/**
	@function setBlog
	
	Sets a blog to use.
	
	@param string	id	Blog ID
	*/
	public function setBlog($id)
	{
		$this->blog = new dcBlog($this, $id);
	}
	
	/**
	@function unsetBlog
	
	Unset blog.
	*/
	public function unsetBlog()
	{
		$this->blog = null;
	}
	
	/**
	@function addFormater
	
	Adds a new text formater which will call the function '''$func''' to
	transform text. The function must be a valid callback, takes one
	argument: the string to transform and returns the transformed string.
	
	@param string	name		Formater name
	@param mixed	func		Function to use, must be a valid callback
	*/
	public function addFormater($name,$func)
	{
		if (is_callable($func)) {
			$this->formaters[$name] = $func;
		}
	}
	
	/**
	@function getFormaters
	
	Return formaters list.
	
	@return array
	*/
	public function getFormaters()
	{
		return array_keys($this->formaters);
	}
	
	/**
	@function callFormater
	
	If '''$name''' is a valid formater, it returns '''$str''' transformed.
	
	@param string	name		Formater name
	@param string	str		String to transform
	*/
	public function callFormater($name,$str)
	{
		if (isset($this->formaters[$name])) {
			return call_user_func($this->formaters[$name],$str);
		}
		
		return $str;
	}
	
	/**
	@function addBehavior
	
	Adds a new behavior to behaviors stack.
	
	@param string	behavior	Behavior name
	@param mixed	func		Function to call. Must be a valid callback
	*/
	public function addBehavior($behavior,$func)
	{
		if (is_callable($func)) {
			$this->behaviors[$behavior][] = $func;
		}
	}
	
	/**
	@function callBehavior
	
	Calls every function in behaviors stack for a given behavior and returns
	concatened result of each function.
	
	@param string	behavior	Behavior name
	@return string
	*/
	public function callBehavior($behavior)
	{
		if (isset($this->behaviors[$behavior]))
		{
			$args = func_get_args();
			array_shift($args);
			
			$res = '';
			
			foreach ($this->behaviors[$behavior] as $f) {
				$res .= call_user_func_array($f,$args);
			}
			
			return $res;
		}
	}
	
	/* ===================================================
	Users management
	=================================================== */
	/** @doc
	=== Users management methods === */
	
	/**
	@function getUser
	
	Returns a user by its ID.
	
	@param string	id		User ID
	@return recordset
	*/
	public function getUser($id)
	{
		$params['user_id'] = $id;
		
		return $this->getUsers($params);
	}
	
	/**
	@function getUsers
	
	Returns a users list. '''$params''' is an array with the following
	optionnal parameters:
	
	 * '''q''': search string (on user_id, user_name, user_firstname)
	 * '''user_id''': user ID
	 * '''order''': ORDER BY clause (default: user_id ASC)
	 * '''limit''': LIMIT clause (should be an array ![limit,offset])
	
	@param array		params		Parameters
	@param boolean		count_only	Only counts results
	@return recordset
	*/
	public function getUsers($params=array(),$count_only=false)
	{
		if ($count_only)
		{
			$strReq =
			'SELECT count(U.user_id) '.
			'FROM '.$this->prefix.'user U '.
			'WHERE NULL IS NULL ';
		}
		else
		{
			$strReq =
			'SELECT U.user_id,user_super,user_pwd,user_name,'.
			'user_firstname,user_displayname,user_email,user_url,'.
			'user_desc, user_lang,user_tz, user_post_status,user_options, '.
			'count(P.post_id) AS nb_post '.
			'FROM '.$this->prefix.'user U '.
				'LEFT JOIN '.$this->prefix.'post P ON U.user_id = P.user_id '.
			'WHERE NULL IS NULL ';
		}
		
		if (!empty($params['q'])) {
			$q = $this->con->escape(str_replace('*','%',strtolower($params['q'])));
			$strReq .= 'AND ('.
				"LOWER(U.user_id) LIKE '".$q."' ".
				"OR LOWER(user_name) LIKE '".$q."' ".
				"OR LOWER(user_firstname) LIKE '".$q."' ".
				') ';
		}
		
		if (!empty($params['user_id'])) {
			$strReq .= "AND U.user_id = '".$this->con->escape($params['user_id'])."' ";
		}
		
		if (!$count_only) {
			$strReq .= 'GROUP BY U.user_id,user_super,user_pwd,user_name,'.
			'user_firstname,user_displayname,user_email,user_url,'.
			'user_desc, user_lang,user_tz,user_post_status,user_options ';
			
			if (!empty($params['order']) && !$count_only) {
				$strReq .= 'ORDER BY '.$this->con->escape($params['order']).' ';
			} else {
				$strReq .= 'ORDER BY U.user_id ASC ';
			}
		}
		
		if (!$count_only && !empty($params['limit'])) {
			$strReq .= $this->con->limit($params['limit']);
		}
		
		$rs = $this->con->select($strReq);
		$rs->extend('rsExtUser');
		return $rs;
	}
	
	/**
	@function addUser
	
	Create a new user. Takes a cursor as input. Returns the new user ID.
	
	@param cursor	cur		User cursor
	@return string
	*/
	public function addUser(&$cur)
	{
		if (!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		if ($cur->user_id == '') {
			throw new Exception(__('No user ID given'));
		}
		
		if ($cur->user_pwd == '') {
			throw new Exception(__('No password given'));
		}
		
		$this->getUserCursor($cur);
		
		if ($cur->user_creadt === null) {
			$cur->user_creadt = array('NOW()');
		}
		
		$cur->insert();
		
		return $cur->user_id;
	}
	
	/**
	@function updUser
	
	Update an existing user. Returns the user ID.
	
	@param string		id		User ID
	@param cursor		cur		User cursor
	@return string
	*/
	public function updUser($id,&$cur)
	{
		$this->getUserCursor($cur);
		
		if (($cur->user_id !== null || $id != $this->auth->userID()) &&
		!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		$cur->update("WHERE user_id = '".$this->con->escape($id)."' ");
		
		if ($cur->user_id !== null) {
			return $cur->user_id;
		}
		
		return $id;
	}
	
	/**
	@function delUser
	
	Delete a user.
	
	@param string		id		User ID
	*/
	public function delUser($id)
	{
		if (!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		$rs = $this->getUser($id);
		
		if ($rs->nb_post == 0)
		{
			$strReq = 'DELETE FROM '.$this->prefix.'user '.
					"WHERE user_id = '".$this->con->escape($id)."' ";
			
			$this->con->execute($strReq);
		}
	}
	
	/**
	@function userExists
	
	Checks if a user exists.
	
	@param string		id		User ID
	@return boolean
	*/
	public function userExists($id)
	{
		$strReq = 'SELECT user_id '.
				'FROM '.$this->prefix.'user '.
				"WHERE user_id = '".$this->con->escape($id)."' ";
		
		$rs = $this->con->select($strReq);
		
		return !$rs->isEmpty();
	}
	
	/**
	@function getUserPermissions
	
	Returns all user permissions as an array which looks like:
	
	 * ![blog_id]
	   * ![permission] => true
	   * ...
	
	@param string		id		User ID
	@return array
	*/
	public function getUserPermissions($id)
	{
		$strReq = 'SELECT blog_id, permissions '.
				'FROM '.$this->prefix.'permissions '.
				"WHERE user_id = '".$this->con->escape($id)."' ";
		
		$rs = $this->con->select($strReq);
		
		$res = array();
		
		while ($rs->fetch())
		{
			$res[$rs->blog_id] = $this->auth->parsePermissions($rs->permissions);
		}
		
		return $res;
	}
	
	/**
	@function setUserPermissions
	
	Set user permissions. The '''$perms''' array looks like:
	
	 * ![blog_id] => '|perm1|perm2|'
	 * ...
	
	@param string		id		User ID
	@param array		perms	Permissions array
	*/
	public function setUserPermissions($id,$perms)
	{
		if (!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		$strReq = 'DELETE FROM '.$this->prefix.'permissions '.
				"WHERE user_id = '".$this->con->escape($id)."' ";
		
		$this->con->execute($strReq);
		
		foreach ($perms as $blog_id => $p) {
			$this->setUserBlogPermissions($id, $blog_id, $p, false);
		}
	}
	
	public function setUserBlogPermissions($id, $blog_id, $perms, $delete_first=true)
	{
		if (!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		$no_perm = empty($perms);
		
		$perms = '|'.implode('|',array_keys($perms)).'|';
		
		$cur = $this->con->openCursor($this->prefix.'permissions');
		
		$cur->user_id = (string) $id;
		$cur->blog_id = (string) $blog_id;
		$cur->permissions = $perms;
		
		if ($delete_first || $no_perm)
		{
			$strReq = 'DELETE FROM '.$this->prefix.'permissions '.
					"WHERE blog_id = '".$this->con->escape($blog_id)."' ".
					"AND user_id = '".$this->con->escape($id)."' ";
			
			$this->con->execute($strReq);
		}
		
		if (!$no_perm) {
			$cur->insert();
		}
	}
	
	public function setUserDefaultBlog($id, $blog_id)
	{
		$cur = $this->con->openCursor($this->prefix.'user');
		
		$cur->user_default_blog = (string) $blog_id;
		
		$cur->update("WHERE user_id = '".$this->con->escape($id)."'");
	}
	
	private function getUserCursor(&$cur)
	{
		if ($cur->isField('user_id')
		&& !preg_match('/^[A-Za-z0-9._-]{2,}$/',$cur->user_id)) {
			throw new Exception(__('User ID must contain at least 2 characters using letters, numbers or symbols.'));
		}
		
		if ($cur->user_url !== null && $cur->user_url != '') {
			if (!preg_match('|^http(s?)://|',$cur->user_url)) {
				$cur->user_url = 'http://'.$cur->user_url;
			}
		}
		
		if ($cur->isField('user_pwd')) {
			if (strlen($cur->user_pwd) < 6) {
				throw new Exception(__('Password must contain at least 6 characters.'));
			}
			$cur->user_pwd = crypt::hmac(DC_MASTER_KEY,$cur->user_pwd);
		}
		
		if ($cur->user_upddt === null) {
			$cur->user_upddt = array('NOW()');
		}
		
		if ($cur->user_options !== null) {
			$cur->user_options = serialize($cur->user_options);
		}
	}
	
	/* ===================================================
	Blogs management
	=================================================== */
	# Get blogs user can access
	public function getUserBlogs()
	{
		$blogs = $this->auth->getPermissions();
		
		foreach ($blogs as $b => $p) {
			if (!$this->auth->check('usage,admin,contentadmin',$b)) {
				unset($blogs[$b]);
			}
		}
		
		$this->blogs = $blogs;
	}
	
	public function getBlog($id)
	{
		$blog = $this->getBlogs(array('blog_id'=>$id));
		
		if ($blog->isEmpty()) {
			return false;
		}
		
		return $blog;
	}
	
	# Get blogs
	public function getBlogs($params=array(),$count_only=false)
	{
		if ($count_only)
		{
			$strReq = 'SELECT count(B.blog_id) '.
					'FROM '.$this->prefix.'blog B '.
					'WHERE NULL IS NULL ';
		}
		else
		{
			$strReq =
			'SELECT B.blog_id, blog_uid, blog_url, blog_name, blog_desc, blog_creadt, '.
			'blog_upddt, blog_status, COUNT(post_id) AS nb_post '.
			'FROM '.$this->prefix.'blog B '.
				'LEFT JOIN '.$this->prefix.'post P ON B.blog_id = P.blog_id '.
			'WHERE NULL IS NULL ';
		}
		
		if (!empty($params['blog_id'])) {
			$strReq .= "AND B.blog_id = '".$this->con->escape($params['blog_id'])."' ";
		}
		
		# If logged in and not super admin, get only user's blogs with status 1 or 0
		if ($this->auth->userID() && !$this->auth->isSuperAdmin()) {
			$inReq = array();
			foreach (array_keys($this->blogs) as $v) {
				$inReq[] = $this->con->escape($v);
			}
			$strReq .=
			"AND B.blog_id IN ('".implode("','",$inReq)."') ".
			"AND blog_status IN (1,0) ";
		} elseif (!$this->auth->userID()) {
			$strReq .= 'AND blog_status = 1 ';
		}
		
		if (!empty($params['q'])) {
			$params['q'] = str_replace('*','%',$params['q']);
			$strReq .=
			"AND (LOWER(B.blog_id) LIKE '".
			$this->con->escape($params['q'])."' ".
			" OR LOWER(B.blog_name) LIKE '".
			$this->con->escape($params['q'])."' ) ";
		}
		
		if (!$count_only) {
			$strReq .= 'GROUP BY B.blog_id, blog_uid, blog_url, blog_name, '.
			'blog_desc, blog_creadt, blog_upddt, blog_status ';
			
			if (!empty($params['order'])) {
				$strReq .= 'ORDER BY '.$this->con->escape($params['order']).' ';
			} else {
				$strReq .= 'ORDER BY B.blog_id ASC ';
			}
		}
		
		if (!$count_only && !empty($params['limit'])) {
			$strReq .= $this->con->limit($params['limit']);
		}
		
		return $this->con->select($strReq);
	}
	
	public function addBlog($cur)
	{
		if (!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		$this->getBlogCursor($cur);
		
		$cur->blog_creadt = date('Y-m-d H:i:s');
		$cur->blog_upddt = date('Y-m-d H:i:s');
		$cur->blog_uid = md5(uniqid());
		
		$cur->insert();
	}
	
	public function updBlog($id,$cur)
	{
		$this->getBlogCursor($cur);
		
		$cur->blog_upddt = date('Y-m-d H:i:s');
		
		$cur->update("WHERE blog_id = '".$this->con->escape($id)."'");
	}
	
	private function getBlogCursor(&$cur)
	{
		if ($cur->blog_id !== null
		&& !preg_match('/^[A-Za-z0-9._-]{2,}$/',$cur->blog_id)) {
			throw new Exception(__('Blog ID must contain at least 2 characters using letters, numbers or symbols.')); 
		}
		
		if ($cur->blog_name !== null && $cur->blog_name == '') {
			throw new Exception(__('No blog name'));
		}
		
		if ($cur->blog_url !== null && $cur->blog_url == '') {
			throw new Exception(__('No blog URL'));
		}
		
		if ($cur->blog_desc !== null) {
			$cur->blog_desc = html::clean($cur->blog_desc);
		}
	}
	
	public function delBlog($id)
	{
		if (!$this->auth->isSuperAdmin()) {
			throw new Exception(__('You are not an administrator'));
		}
		
		$strReq = 'DELETE FROM '.$this->prefix.'blog '.
				"WHERE blog_id = '".$this->con->escape($id)."' ";
		
		$this->con->execute($strReq);
	}
	
	public function blogExists($id)
	{
		$strReq = 'SELECT blog_id '.
				'FROM '.$this->prefix.'blog '.
				"WHERE blog_id = '".$this->con->escape($id)."' ";
		
		$rs = $this->con->select($strReq);
		
		return !$rs->isEmpty();
	}
	
	/* ===================================================
	Dashboard methods
	=================================================== */
	public function getLastPosts()
	{
		$strReq = 'SELECT post_title, post_id, post_dt, blog_id '.
				'FROM '.$this->prefix.'post '.
				'WHERE NULL IS NULL ';
		
		$blog_ids = array();
		foreach ($this->blogs as $k => $v) {
			$blog_ids[] = "blog_id = '".$this->con->escape($k)."' ";
		}
		
		if (!empty($blog_ids)) {
			$strReq .= 'AND ('.implode(' OR ',$blog_ids).') ';
		}
		
		$strReq .= 'ORDER BY post_dt DESC ';
		$strReq .= 'LIMIT 0,50';
		
		$rs = $this->con->select($strReq);
		
		while ($rs->fetch()) {
			$rs->blog_name = $this->blogs[$rs->blog_id]['name'];
		}
		
		return $rs;
	}
	
	/* ===================================================
	HTML Filters
	=================================================== */
	public function HTMLfilter($str)
	{
		$filter = new htmlFilter;
		$str = trim($filter->apply($str));
		return $str;
	}
	
	/* ===================================================
	Wiki2xhtml options
	=================================================== */
	private function initWiki()
	{
		$this->wiki2xhtml = new wiki2xhtml;
	}
	
	public function wikiTransform($str)
	{
		if (!($this->wiki2xhtml instanceof wiki2xhtml)) {
			$this->initWiki();
		}
		return $this->wiki2xhtml->transform($str);
	}
	
	public function initWikiPost()
	{
		$this->initWiki();
		
		$this->wiki2xhtml->setOpts(array(
			'active_title' => 1,
			'active_setext_title' => 0,
			'active_hr' => 1,
			'active_lists' => 1,
			'active_quote' => 1,
			'active_pre' => 1,
			'active_empty' => 1,
			'active_auto_urls' => 0,
			'active_urls' => 1,
			'active_auto_img' => 0,
			'active_img' => 1,
			'active_anchor' => 1,
			'active_em' => 1,
			'active_strong' => 1,
			'active_br' => 1,
			'active_q' => 1,
			'active_code' => 1,
			'active_acronym' => 1,
			'active_ins' => 1,
			'active_del' => 1,
			'active_footnotes' => 1,
			'active_wikiwords' => 0,
			'active_macros' => 1,
			'parse_pre' => 1,
			'active_fr_syntax' => 0,
			'first_title_level' => 3,
			'note_prefix' => 'wiki-footnote',
			'note_str' => '<div class="footnotes"><h4>Notes</h4>%s</div>'
		));
		
		# --BEHAVIOR-- coreWikiPostInit
		$this->callBehavior('coreInitWikiPost',$this->wiki2xhtml);
	}
	
	public function initWikiSimpleComment()
	{
		$this->initWiki();
		
		$this->wiki2xhtml->setOpts(array(
			'active_title' => 0,
			'active_setext_title' => 0,
			'active_hr' => 0,
			'active_lists' => 0,
			'active_quote' => 0,
			'active_pre' => 0,
			'active_empty' => 0,
			'active_auto_urls' => 1,
			'active_urls' => 0,
			'active_auto_img' => 0,
			'active_img' => 0,
			'active_anchor' => 0,
			'active_em' => 0,
			'active_strong' => 0,
			'active_br' => 0,
			'active_q' => 0,
			'active_code' => 0,
			'active_acronym' => 0,
			'active_ins' => 0,
			'active_del' => 0,
			'active_footnotes' => 0,
			'active_wikiwords' => 0,
			'active_macros' => 0,
			'parse_pre' => 0,
			'active_fr_syntax' => 0
		));
		
		# --BEHAVIOR-- coreInitWikiSimpleComment
		$this->callBehavior('coreInitWikiSimpleComment',$this->wiki2xhtml);
	}
	
	public function initWikiComment()
	{
		$this->initWiki();
		
		$this->wiki2xhtml->setOpts(array(
			'active_title' => 0,
			'active_setext_title' => 0,
			'active_hr' => 0,
			'active_lists' => 1,
			'active_quote' => 0,
			'active_pre' => 1,
			'active_empty' => 0,
			'active_auto_urls' => 1,
			'active_urls' => 1,
			'active_auto_img' => 0,
			'active_img' => 0,
			'active_anchor' => 0,
			'active_em' => 1,
			'active_strong' => 1,
			'active_br' => 1,
			'active_q' => 1,
			'active_code' => 1,
			'active_acronym' => 1,
			'active_ins' => 1,
			'active_del' => 1,
			'active_footnotes' => 0,
			'active_wikiwords' => 0,
			'active_macros' => 0,
			'parse_pre' => 0,
			'active_fr_syntax' => 0
		));
		
		# --BEHAVIOR-- coreInitWikiComment
		$this->callBehavior('coreInitWikiComment',$this->wiki2xhtml);
	}
	
	public function blogDefaults($defaults=null)
	{
		#comment_notification
		#show_previews
		if (!is_array($defaults))
		{
			$defaults = array(
				array('allow_comments','boolean',true,
				'Allow comments on blog'),
				array('allow_trackbacks','boolean',true,
				'Allow trackbacks on blog'),
				array('blog_timezone','string','Europe/London',
				'Blog timezone'),
				array('comments_nofollow','boolean',true,
				'Add rel="nofollow" to comments URLs'),
				array('comments_pub','boolean',true,
				'Publish comments immediatly'),
				array('comments_ttl','integer',0,
				'Number of days to keep comments and trackbacks open (0 means no ttl)'),
				array('copyright_notice','string','','Copyright notice (simple text)'),
				array('date_format','string','%A, %B %e %Y',
				'Date format. See PHP strftime function for patterns'),
				array('editor','string','',
				'Person responsible of the content'),
				array('lang','string','en',
				'Default blog language'),
				array('nb_post_per_page','integer',20,
				'Number of entries on home page and category pages'),
				array('post_url_format','string','{y}/{m}/{d}/{t}',
				'Post URL format. {y}: year, {m}: month, {d}: day, {id}: post id, {t}: entry title'),
				array('public_path','string','public',
				'Path to public directory, begins with a / for a full system path'),
				array('public_url','string','/public',
				'URL to public directory'),
				array('theme','string','default',
				'Blog theme'),
				array('themes_path','string','themes',
				'Themes root path'),
				array('themes_url','string','/themes',
				'Themes root URL'),
				array('time_format','string','%H:%M',
				'Time format. See PHP strftime function for patterns'),
				array('url_scan','string','path_info',
				'URL handle mode (path_info or query_string)'),
				array('use_smilies','boolean',false,
				'Show smilies on entries and comments'),
				array('wiki_comments','boolean',false,
				'Allow commenters to use a subset of wiki syntax')
			);
		}
		
		$settings = new dcSettings($this,null);
		$settings->setNameSpace('system');
		
		foreach ($defaults as $v) {
			$settings->put($v[0],$v[2],$v[1],$v[3],false,true);
		}
	}
	
	public function userDefaults()
	{
		return array(
			'edit_size' => 15,
			'enable_wysiwyg' => true,
			'post_format' => 'wiki',
		);
	}
	
	/* ===================================================
	Maintenance methods
	=================================================== */
	function indexAllPosts($start=null,$limit=null)
	{
		$strReq = 'SELECT COUNT(post_id) '.
				'FROM '.$this->prefix.'post';
		$rs = $this->con->select($strReq);
		$count = $rs->f(0);
		
		$strReq = 'SELECT post_id, post_title, post_excerpt_xhtml, post_content_xhtml '.
				'FROM '.$this->prefix.'post ';
		
		if ($start !== null && $limit !== null) {
			$strReq .= $this->con->limit($start,$limit);
		}
		
		$rs = $this->con->select($strReq,true);
		
		$cur = $this->con->openCursor($this->prefix.'post');
		
		while ($rs->fetch())
		{
			$words = $rs->post_title.' '.	$rs->post_excerpt_xhtml.' '.
			$rs->post_content_xhtml;
			
			$cur->post_words = implode(' ',text::splitWords($words));
			$cur->update('WHERE post_id = '.(integer) $rs->post_id);
			$cur->clean();
		}
		
		if ($start+$limit > $count) {
			return null;
		}
		return $start+$limit;
	}
	
	function indexAllComments($start=null,$limit=null)
	{
		$strReq = 'SELECT COUNT(comment_id) '.
				'FROM '.$this->prefix.'comment';
		$rs = $this->con->select($strReq);
		$count = $rs->f(0);
		
		$strReq = 'SELECT comment_id, comment_content '.
				'FROM '.$this->prefix.'comment ';
		
		if ($start !== null && $limit !== null) {
			$strReq .= $this->con->limit($start,$limit);
		}
		
		$rs = $this->con->select($strReq);
		
		$cur = $this->con->openCursor($this->prefix.'comment');
		
		while ($rs->fetch())
		{
			$cur->comment_words = implode(' ',text::splitWords($rs->comment_content));
			$cur->update('WHERE comment_id = '.(integer) $rs->comment_id);
			$cur->clean();
		}
		
		if ($start+$limit > $count) {
			return null;
		}
		return $start+$limit;
	}
	
	function countAllComments()
	{
		$strReq = 'SELECT COUNT(comment_id) AS nb, post_id '.
				'FROM '.$this->prefix.'comment '.
				'WHERE comment_trackback %s 1 '.
				'AND comment_status = 1 '.
				'GROUP BY post_id ';
		
		$rsC = $this->con->select(sprintf($strReq,'<>'));
		$rsT = $this->con->select(sprintf($strReq,'='));
		
		$cur = $this->con->openCursor($this->prefix.'post');
		while ($rsC->fetch()) {
			$cur->nb_comment = (integer) $rsC->nb;
			$cur->update('WHERE post_id = '.(integer) $rsC->post_id);
			$cur->clean();
		}
		
		while ($rsT->fetch()) {
			$cur->nb_trackback = (integer) $rsT->nb;
			$cur->update('WHERE post_id = '.(integer) $rsT->post_id);
			$cur->clean();
		}
	}
}
?>