<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of DotClear.
# Copyright (c) 2005 Olivier Meunier. All rights
# reserved.
#
# DotClear is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# DotClear is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with DotClear; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****

require dirname(__FILE__).'/../ext/incutio.ixr_library.php';

class dcXmlRpc extends IXR_IntrospectionServer
{
	private $core;
	private $debug = false;
	private $debug_file = '/tmp/dotclear-xmlrpc.log';
	
	public function __construct(&$core)
	{
		$this->IXR_IntrospectionServer();
		
		$this->core =& $core;
		
		# Blogger methods
		$this->addCallback('blogger.newPost','this:blogger_newPost',
			array('string','string','string','string','string','string','integer'),
			'New post');
		
		$this->addCallback('blogger.editPost','this:blogger_editPost',
			array('boolean','string','string','string','string','string','integer'),
			'Edit a post');
		
		$this->addCallback('blogger.getPost','this:blogger_getPost',
			array('struct','string','integer','string','string'),
			'Return a posts by ID');
		
		$this->addCallback('blogger.deletePost','this:blogger_deletePost',
			array('string','string','string','string','string','integer'),
			'Delete a post');
		
		$this->addCallback('blogger.getRecentPosts','this:blogger_getRecentPosts',
			array('array','string','string','string','string','integer'),
			'Return a list of recent posts');
		
		$this->addCallback('blogger.getUsersBlogs','this:blogger_getUserBlogs',
			array('struct','string','string','string'),
			"Return user's blog");
		
		$this->addCallback('blogger.getUserInfo','this:blogger_getUserInfo',
			array('struct','string','string','string'),
			'Return User Info');
		
		# Metaweblog methods
		$this->addCallback('metaWeblog.newPost','this:mw_newPost',
			array('string','string','string','string','struct','boolean'),
			'Creates a new post, and optionnaly publishes it.');
		
		$this->addCallback('metaWeblog.editPost','this:mw_editPost',
			array('boolean','string','string','string','struct','boolean'),
			'Updates information about an existing entry');
		
		$this->addCallback('metaWeblog.getPost','this:mw_getPost',
			array('struct','string','string','string'),
			'Returns information about a specific post');
		
		$this->addCallback('metaWeblog.getRecentPosts','this:mw_getRecentPosts',
			array('array','string','string','string','integer'),
			'List of most recent posts in the system');
		
		$this->addCallback('metaWeblog.newMediaObject','this:mw_newMediaObject',
			array('string','string','string','string','struct'),
			'Download a file on the web server');
		
		# MovableType methods
		$this->addCallback('mt.getRecentPostTitles','this:mt_getRecentPostTitles',
			array('array','string','string','string','integer'),
			'List of most recent posts in the system');
		
		$this->addCallback('mt.getCategoryList','this:mt_getCategoryList',
			array('array','string','string','string'),
			'List of all categories defined in the weblog');
		
		$this->addCallback('mt.getPostCategories','this:mt_getPostCategories',
			array('array','string','string','string'),
			'List of all categories to which the post is assigned');
		
		$this->addCallback('mt.setPostCategories','this:mt_setPostCategories',
			array('boolean','string','string','string','array'),
			'Sets the categories for a post');
		
		$this->addCallback('mt.publishPost','this:mt_publishPost',
			array('boolean','string','string','string'),
			'Retrieve pings list for a post');
		
		$this->addCallback('mt.supportedMethods','this:listMethods',
			array(),'Retrieve information about the XML-RPC methods supported by the server.');
	}
	
	public function serve($data=false,$encoding='UTF-8')
	{
		parent::serve(false,$encoding);
	}
	
	private function debugTrace($function,$msg)
	{
		if ($this->debug)
		{
			if (($fp = @fopen($this->debug_file,'a')) !== false)
			{
				if (is_array($msg)) {
					$msg = implode(' : ',$msg);
				}
				
				fwrite($fp,'['.date('r').']'.' '.$function.' - '.$msg."\n");
				fclose($fp);
			}
		}
	}
	
	
	/* Internal methods
	--------------------------------------------------- */
	private function setUser($user_id,$pwd)
	{
		if ($this->core->auth->checkUser($user_id,$pwd) !== true) {
			throw new Exception('Login error');
		}
		
		$this->core->getUserBlogs();
		return true;
	}
	
	private function setBlog($blog_id)
	{
		$this->core->setBlog($blog_id);
		if (!$this->core->auth->check('usage,contentadmin',$this->core->blog->id)) {
			$this->core->blog = null;
			throw new Exception('This blog does not exists');
		}
		
		return true;
	}
	
	private function setBlogByPost($post_id)
	{
		$strReq = 'SELECT blog_id '.
				'FROM '.$this->core->prefix.'post '.
				'WHERE post_id = '.(integer) $post_id;
		
		$rs = $this->core->con->select($strReq);
		
		if ($rs->isEmpty()) {
			throw new Exception('This post does not exist');
		}
		
		$this->setBlog($rs->blog_id);
	}
	
	private function getPostRS($post_id,$user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlogByPost($post_id);
			$rs = $this->core->blog->getPosts(array('post_id' => (integer) $post_id));
		} catch (Exception $e) {
			throw $e;
		}
		
		if ($rs->isEmpty()) {
			throw new Exception('This entry does not exist');
		}
		
		return $rs;
	}
	
	/* Generic methods
	--------------------------------------------------- */
	private function newPost($blog_id,$user,$pwd,$content,$struct=array(),$publish=true)
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog($blog_id);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$title = !empty($struct['title']) ? $struct['title'] : '';
		$excerpt = !empty($struct['mt_excerpt']) ? $struct['mt_excerpt'] : '';
		$description = !empty($struct['description']) ? $struct['description'] : null;
		$dateCreated = !empty($struct['dateCreated']) ? $struct['dateCreated'] : null;
		$open_comment = isset($struct['mt_allow_comments']) ? $struct['mt_allow_comments'] : 1;
		$open_tb = isset($struct['mt_allow_pings']) ? $struct['mt_allow_pings'] : 1;
		
		if ($description !== null) {
			$content = $description;
		}
		
		if (!$title) {
			$title = text::cutString(html::clean($content),25).'...';
		}
		
		$excerpt_xhtml = $this->core->callFormater('xhtml',$excerpt);
		$content_xhtml = $this->core->callFormater('xhtml',$content);
		
		if (empty($content)) {
			return new IXR_Error(804,'Cannot create an empty entry');
		}
		
		$cur = $this->core->con->openCursor($this->core->prefix.'post');
		
		$cur->user_id = $this->core->auth->userID();
		$cur->post_lang = $this->core->auth->getInfo('user_lang');
		$cur->post_title = trim($title);
		$cur->post_content = $content;
		$cur->post_excerpt = $excerpt;
		$cur->post_content_xhtml = $content_xhtml;
		$cur->post_excerpt_xhtml = $excerpt_xhtml;
		$cur->post_open_comment = $open_comment;
		$cur->post_open_tb = $open_tb;
		$cur->post_status = (integer) $publish;
		$cur->post_format = 'xhtml';
		
		if ($dateCreated) {
			$cur->post_dt = date('Y-m-d H:i:00',strtotime($dateCreated));
		}
		
		try {
			$post_id = $this->core->blog->addPost($cur);
		} catch (Exception $e) {
			return new IXR_Error(805,$e->getMessage());
		}
		
		return $post_id;
	}
	
	private function editPost($post_id,$user,$pwd,$content,$struct=array(),$publish=true)
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$title = (!empty($struct['title'])) ? $struct['title'] : '';
		$excerpt = (!empty($struct['mt_excerpt'])) ? $struct['mt_excerpt'] : '';
		$description = (!empty($struct['description'])) ? $struct['description'] : null;
		$dateCreated = !empty($struct['dateCreated']) ? $struct['dateCreated'] : null;
		$open_comment = (isset($struct['mt_allow_comments'])) ? $struct['mt_allow_comments'] : 1;
		$open_tb = (isset($struct['mt_allow_pings'])) ? $struct['mt_allow_pings'] : 1;
		
		if ($description !== null) {
			$content = $description;
		}
		
		if (!$title) {
			$title = text::cutString(html::clean($content),25).'...';
		}
		
		$excerpt_xhtml = $this->core->callFormater('xhtml',$excerpt);
		$content_xhtml = $this->core->callFormater('xhtml',$content);
		
		if (empty($content)) {
			return new IXR_Error(804,'Cannot create an empty entry');
		}
		
		$cur = $this->core->con->openCursor($this->core->prefix.'post');
		
		$cur->post_title = trim($title);
		$cur->post_content = $content;
		$cur->post_excerpt = $excerpt;
		$cur->post_content_xhtml = $content_xhtml;
		$cur->post_excerpt_xhtml = $excerpt_xhtml;
		$cur->post_open_comment = $open_comment;
		$cur->post_open_tb = $open_tb;
		$cur->post_status = (integer) $publish;
		$cur->post_format = 'xhtml';
		$cur->post_url = $post->post_url;
		
		if ($dateCreated) {
			$cur->post_dt = date('Y-m-d H:i:00',strtotime($dateCreated));
		}
		
		try {
			$post_id = $this->core->blog->updPost($post_id,$cur);
		} catch (Exception $e) {
			return new IXR_Error(805,$e->getMessage());
		}
		
		return true;
	}
	
	private function getPost($post_id,$user,$pwd,$type='mw')
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$res = array();
		
		$res['dateCreated'] = new IXR_Date($post->getTS());
		$res['userid'] = $post->user_id;
		$res['postid'] = $post->post_id;
		
		if ($type == 'blogger') {
			$res['content'] = $post->post_content_xhtml;
		}
		
		if ($type == 'mt' || $type == 'mw') {
			$res['title'] = $post->post_title;
		}
		
		if ($type == 'mw') {
			$res['description'] = $post->post_content_xhtml;
			$res['link'] = $res['permaLink'] = $post->getURL();
			$res['mt_excerpt'] = $post->post_excerpt_xhtml;
			$res['mt_text_more'] = '';
			$res['mt_allow_comments'] = (integer) $post->post_open_comment;
			$res['mt_allow_pings'] = (integer) $post->post_open_tb;
			$res['mt_convert_breaks'] = '';
			$res['mt_keywords'] = '';
		}
		
		return $res;
	}
	
	private function deletePost($post_id,$user,$pwd)
	{
		$post_id = (integer) $post_id;
		
		try {
			$this->setUser($user,$pwd);
			$this->setBlogByPost($post_id);
			$this->core->blog->delPost($post_id);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return true;
	}
	
	private function getRecentPosts($blog_id,$user,$pwd,$nb_post,$type='mw')
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog($blog_id);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$nb_post = (integer) $nb_post;
		
		if ($nb_post > 50) {
			return new IXR_Error(805,'Cannot retrieve more than 50 entries');
		}
		
		$params = array();
		$params['limit'] = $nb_post;
		
		try {
			$posts = $this->core->blog->getPosts($params);
		} catch (Exception $e) {
			return new IXR_Error(805,$e->getMessage());
		}
		
		$res = array();
		while ($posts->fetch())
		{
			$tres = array();
			
			$tres['dateCreated'] = new IXR_Date($posts->getTS());
			$tres['userid'] = $posts->user_id;
			$tres['postid'] = $posts->post_id;
			
			if ($type == 'blogger') {
				$tres['content'] = $posts->post_content_xhtml;
			}
			
			if ($type == 'mt' || $type == 'mw') {
				$tres['title'] = $posts->post_title;
			}
			
			if ($type == 'mw') {
				$tres['description'] = $posts->post_content_xhtml;
				$tres['link'] = $tres['permaLink'] = $posts->getURL();
				$tres['mt_excerpt'] = $posts->post_excerpt_xhtml;
				$tres['mt_text_more'] = '';
				$tres['mt_allow_comments'] = (integer) $posts->post_open_comment;
				$tres['mt_allow_pings'] = (integer) $posts->post_open_tb;
				$tres['mt_convert_breaks'] = '';
				$tres['mt_keywords'] = '';
			}
			
			$res[] = $tres;
		}
		
		return $res;
	}
	
	private function getUsersBlogs($user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$res = array();
		foreach ($this->core->blogs as $k => $v) {
			$res[] = array(
				'url' => $v['url'],
				'blogid' => $k,
				'blogName' => $v['name']
			);
		}
		return $res;
	}
	
	private function getUserInfo($user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return array(
			'userid' => $this->core->auth->userID(),
			'firstname' => $this->core->auth->getInfo('user_firstname'),
			'lastname' => $this->core->auth->getInfo('user_name'),
			'nickname' => $this->core->auth->getInfo('user_displayname'),
			'email' => $this->core->auth->getInfo('user_email'),
			'url' => $this->core->auth->getInfo('user_url')
		);
	}
	
	private function getCategoryList($blog_id,$user,$pwd)
	{
		try {
			$this->setUser($user,$pwd);
			$this->setBlog($blog_id);
			$rs = $this->core->blog->getCategories();
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$res = array();
		while ($rs->fetch()) {
			$res[] = array(
				'categoryId' => $rs->cat_id,
				'categoryName' => $rs->cat_title
			);
		}
		
		return $res;
	}
	
	private function getPostCategories($post_id,$user,$pwd)
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return array(
			array(
			'categoryName' => $post->cat_title,
			'categoryId' => $post->cat_id,
			'isPrimary' => true
			)
		);
		
		return $res;
	}
	
	private function setPostCategories($post_id,$user,$pwd,$categories)
	{
		$post_id = (integer) $post_id;
		
		try {
			$post = $this->getPostRS($post_id,$user,$pwd);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		$cat_id = (!empty($categories[0]['categoryId'])) ? $categories[0]['categoryId'] : NULL;
		
		foreach($categories as $v)
		{
			if (isset($v['isPrimary']) && $v['isPrimary']) {
				$cat_id = $v['categoryId'];
				break;
			}
		}
		
		try {
			$this->core->blog->updPostCategory($post_id,(integer) $cat_id);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return true;
	}
	
	private function publishPost($post_id,$user,$pwd)
	{
		$post_id = (integer) $post_id;
		
		try {
			$this->setUser($user,$pwd);
			$this->setBlogByPost($post_id);
			$this->core->blog->updPostStatus($post_id,1);
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
		
		return true;
	}
	
	private function newMediaObject($blog_id,$user,$pwd,$file)
	{
		if (empty($file['name'])) {
			return new IXR_Error(801,'No file name');
		}
		
		if (empty($file['bits'])) {
			return new IXR_Error(801,'No file content');
		}
		
		$file_name = $file['name'];
		$file_bits = base64_decode($file['bits']);
		
		try {
			$this->setUser($user,$pwd);
			$this->setBlog($blog_id);
			
			$media = new dcMedia($this->core);
			$media_id = $media->uploadBits($file_name,$file_bits);
			
			$f = $media->getFile($media_id);
			return $f->file_url;
		} catch (Exception $e) {
			return new IXR_Error(801,$e->getMessage());
		}
	}
	
	/* Blogger methods
	--------------------------------------------------- */
	public function blogger_newPost($args)
	{
		$this->debugTrace('blogger.newPost',$args);
		return $this->newPost($args[1],$args[2],$args[3],$args[4],array(),$args[5]);
	}
	
	public function blogger_editPost($args)
	{
		$this->debugTrace('blogger.editPost',$args);
		return $this->editPost($args[1],$args[2],$args[3],$args[4],array(),$args[5]);
	}
	
	public function blogger_getPost($args)
	{
		$this->debugTrace('blogger.getPost',$args);
		return $this->getPost($args[1],$args[2],$args[3],'blogger');
	}
	
	public function blogger_deletePost($args)
	{
		$this->debugTrace('blogger.deletePost',$args);
		return $this->deletePost($args[1],$args[2],$args[3]);
	}
	
	public function blogger_getRecentPosts($args)
	{
		$this->debugTrace('blogger.getRecentPosts',$args);
		return $this->getRecentPosts($args[1],$args[2],$args[3],$args[4],'blogger');
	}
	
	public function blogger_getUserBlogs($args)
	{
		$this->debugTrace('blogger.getUsersBlogs',$args);
		return $this->getUsersBlogs($args[1],$args[2]);
	}
	
	public function blogger_getUserInfo($args)
	{
		$this->debugTrace('blogger.getUserInfo',$args);
		return $this->getUserInfo($args[1],$args[2]);
	}
	
	
	/* Metaweblog methods
	------------------------------------------------------- */
	public function mw_newPost($args)
	{
		$this->debugTrace('metaweblog.newPost',$args);
		return $this->newPost($args[0],$args[1],$args[2],'',$args[3],$args[4]);
	}
	
	public function mw_editPost($args)
	{
		$this->debugTrace('metaweblog.editPost',$args);
		return $this->editPost($args[0],$args[1],$args[2],'',$args[3],$args[4]);
	}
	
	public function mw_getPost($args)
	{
		$this->debugTrace('metaweblog.getPost',$args);
		return $this->getPost($args[0],$args[1],$args[2],'mw');
	}
	
	public function mw_getRecentPosts($args)
	{
		$this->debugTrace('metaweblog.getRecentPosts',$args);
		return $this->getRecentPosts($args[0],$args[1],$args[2],$args[3],'mw');
	}
	
	public function mw_newMediaObject($args)
	{
		$this->debugTrace('metaweblog.newMediaObject',$args);
		return $this->newMediaObject($args[0],$args[1],$args[2],$args[3]);
	}
	
	/* MovableType methods
	--------------------------------------------------- */
	public function mt_getRecentPostTitles($args)
	{
		$this->debugTrace('mt.getRecentPostTitles',$args);
		return $this->getRecentPosts($args[0],$args[1],$args[2],$args[3],'mt');
	}
	
	public function mt_getCategoryList($args)
	{
		$this->debugTrace('mt.getCategoryLis',$args);
		return $this->getCategoryList($args[0],$args[1],$args[2]);
	}
	
	public function mt_getPostCategories($args)
	{
		$this->debugTrace('mt.getPostCategories',$args);
		return $this->getPostCategories($args[0],$args[1],$args[2]);
	}
	
	public function mt_setPostCategories($args)
	{
		$this->debugTrace('mt.setPostCategories',$args);
		return $this->setPostCategories($args[0],$args[1],$args[2],$args[3]);
	}
	
	public function mt_publishPost($args)
	{
		$this->debugTrace('mt.publishPost',$args);
		return $this->publishPost($args[0],$args[1],$args[2]);
	}
}
?>