<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of DotClear.
# Copyright (c) 2005 Olivier Meunier and contributors. All rights
# reserved.
#
# DotClear is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# DotClear is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with DotClear; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****

class dcBlog
{
	public $con;
	public $prefix;
	
	public $id;
	public $uid;
	public $name;
	public $desc;
	public $url;
	public $host;
	public $creadt;
	public $upddt;
	
	public $user;
	
	public $settings;
	public $themes_path;
	public $public_path;
	
	private $post_status = array();
	private $comment_status = array();
	
	public $without_password;
	
	public function __construct(&$core, $id)
	{
		$this->con =& $core->con;
		$this->prefix = $core->prefix;
		$this->core =& $core;
		
		$this->without_password = true;
		
		if (($b = $this->core->getBlog($id)) !== false)
		{
			$this->id = $id;
			$this->uid = $b->blog_uid;
			$this->name = $b->blog_name;
			$this->desc = $b->blog_desc;
			$this->url = $b->blog_url;
			$this->host = preg_replace('|^([a-z]{3,}://)(.*?)/.*$|','$1$2',$this->url);
			$this->creadt = strtotime($b->blog_creadt);
			$this->upddt = strtotime($b->blog_upddt);
			
			$this->settings = new dcSettings($this->core,$this->id);
			
			$this->themes_path = path::fullFromRoot($this->settings->themes_path,DC_ROOT);
			$this->public_path = path::fullFromRoot($this->settings->public_path,DC_ROOT);
			
			$this->post_status['-2'] = __('pending');
			$this->post_status['-1'] = __('scheduled');
			$this->post_status['0'] = __('unpublished');
			$this->post_status['1'] = __('published');
			
			$this->comment_status['-2'] = __('junk');
			$this->comment_status['-1'] = __('pending');
			$this->comment_status['0'] = __('unpublished');
			$this->comment_status['1'] = __('published');
			
			# --BEHAVIOR-- coreBlogConstruct
			$this->core->callBehavior('coreBlogConstruct',$this);
		}
	}
	
	public function getPostStatus($s)
	{
		if (isset($this->post_status[$s])) {
			return $this->post_status[$s];
		}
		return $this->post_status['0'];
	}
	
	public function getAllPostStatus()
	{
		return $this->post_status;
	}
	
	public function getAllCommentStatus()
	{
		return $this->comment_status;
	}
	
	public function withoutPassword($v)
	{
		$this->without_password = (boolean) $v;
	}
	
	/* ===================================================
	Triggers
	=================================================== */
	public function triggerBlog()
	{
		$cur = $this->con->openCursor($this->prefix.'blog');
		
		$cur->blog_upddt = date('Y-m-d H:i:s');
		
		try {
			$cur->update("WHERE blog_id = '".$this->con->escapeStr($this->id)."' ");
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function triggerComment($id,$del=false)
	{
		$id = (integer) $id;
		
		$strReq = 'SELECT post_id, comment_trackback '.
				'FROM '.$this->prefix.'comment '.
				'WHERE comment_id = '.$id.' ';
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		$post_id = $rs->post_id;
		$tb = (boolean) $rs->comment_trackback;
		
		$strReq = 'SELECT COUNT(post_id) '.
				'FROM '.$this->prefix.'comment '.
				'WHERE post_id = '.(integer) $post_id.' '.
				'AND comment_trackback = '.(integer) $tb.' '.
				'AND comment_status = 1 ';
		
		if ($del) {
			$strReq .= 'AND comment_id <> '.$id.' ';
		}
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		$cur = $this->con->openCursor($this->prefix.'post');
		
		if ($rs->isEmpty()) {
			return;
		}
		
		if ($tb) {
			$cur->nb_trackback = (integer) $rs->f(0);
		} else {
			$cur->nb_comment = (integer) $rs->f(0);
		}
		
		try {
			$cur->update('WHERE post_id = '.(integer) $post_id);
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	/* ===================================================
	Categories
	=================================================== */
	/** @doc
	=== Categories management methods === */
	
	public function getCategories($params=array())
	{
		$and = '';
		
		if (!$this->core->auth->userID()) {
			$and = 'AND P.post_status = 1 ';
		}
		
		$strReq =
		'SELECT  C.cat_id, cat_title, cat_url, cat_desc, '.
		'cat_position, COUNT(P.post_id) AS nb_post '.
		'FROM '.$this->prefix.'category C LEFT JOIN '.
		$this->prefix.'post P ON C.cat_id = P.cat_id '.
		"WHERE C.blog_id = '".$this->con->escapeStr($this->id)."' ".
		$and;
		
		if (!empty($params['cat_url'])) {
			$strReq .= "AND cat_url = '".$this->con->escapeStr($params['cat_url'])."' ";
		}
		
		$strReq .= 'GROUP BY C.cat_id, cat_title, cat_url, cat_desc, cat_position ';
		
		if (!empty($params['order'])) {
			$strReq .= 'ORDER BY '.$this->con->escapeStr($params['order']).' ';
		} else {
			$strReq .= 'ORDER BY cat_position ASC ';
		}
		
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		return $rs;
	}
	
	public function getCategory($id)
	{
		$strReq = 'SELECT  C.cat_id, cat_title, cat_url, cat_desc, '.
				'cat_position, COUNT(P.post_id) AS nb_post '.
				'FROM '.$this->prefix.'category C LEFT JOIN '.
				$this->prefix.'post P ON C.cat_id = P.cat_id '.
				"WHERE C.blog_id = '".$this->con->escapeStr($this->id)."' ".
				'AND C.cat_id = '.(integer) $id.' '.
				'GROUP BY C.cat_id, cat_title, cat_url, cat_desc, cat_position ';
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		return $rs;
	}
	
	public function addCategory($cur)
	{
		if (!$this->core->auth->check('categories',$this->id)) {
			throw new Exception(__('You are not allowed to add categories'));
		}
		
		try {
			$this->getCategoryCursor($cur);
		} catch (Exception $e) {
			throw $e;
		}
		
		# Get ID
		try {
			$rs = $this->con->select(
				'SELECT MAX(cat_id) AS max_cat_id, '.
				'MAX(cat_position) AS max_position '.
				'FROM '.$this->prefix.'category ' 
			);
		} catch (Exception $e) {
			throw $e;
		}
		
		$cur->cat_id = (integer) $rs->f(0) + 1;
		$cur->cat_position = $rs->f(1) + 1;
		$cur->blog_id = (string) $this->id;
		
		try {
			$cur->insert();
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
		
		return $cur->cat_id;
	}
	
	public function updCategory($id,&$cur)
	{
		if (!$this->core->auth->check('categories',$this->id)) {
			throw new Exception(__('You are not allowed to update categories'));
		}
		
		try {
			$this->getCategoryCursor($cur,$id);
		} catch (Exception $e) {
			throw $e;
		}
		
		try {
			$cur->update(
			'WHERE cat_id = '.(integer) $id.' '.
			"AND blog_id = '".$this->con->escapeStr($this->id)."' ");
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function updCategoryOrder($id,$order)
	{
		$cur = $this->con->openCursor($this->prefix.'category');
		
		$cur->cat_position = (integer) $order;
		
		try {
			$cur->update(
			'WHERE cat_id = '.(integer) $id.' '.
			"AND blog_id = '".$this->con->escapeStr($this->id)."' ");
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function delCategory($id)
	{
		if (!$this->core->auth->check('categories',$this->id)) {
			throw new Exception(__('You are not allowed to delete categories'));
		}
		
		$strReq = 'SELECT post_id '.
				'FROM '.$this->prefix.'post '.
				'WHERE cat_id = '.(integer) $id.' '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ";
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		if (!$rs->isEmpty()) {
			throw new Exception(__('This category is not empty.'));
		}
		
		$strReq = 'DELETE FROM '.$this->prefix.'category '.
				'WHERE cat_id = '.(integer) $id.' '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ";
		
		try {
			$this->con->execute($strReq);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	private function checkCategory($title,$url,$id=null)
	{
		$strReq = 'SELECT cat_id '.
				'FROM '.$this->prefix.'category '.
				"WHERE (cat_title = '".$this->con->escapeStr($title)."' ".
				"OR cat_url = '".$this->con->escapeStr($url)."') ".
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ";
		
		if ($id !== null) {
			$strReq .= 'AND cat_id <> '.(integer) $id.' ';
		}
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		if (!$rs->isEmpty()) {
			throw new Exception(__('Category title and URL must be unique.'));
		}
	}
	
	private function getCategoryCursor(&$cur,$id=null)
	{
		if ($cur->cat_title == '') {
			throw new Exception(__('You must provide a category title'));
		}
		
		# If we don't have any cat_url, let's do one
		if ($cur->cat_url == '') {
			$cur->cat_url = text::str2URL($cur->cat_title,false);
		}
		
		# Still empty ?
		if ($cur->cat_url == '') {
			throw new Exception(__('You must provide a category URL'));
		} else {
			$cur->cat_url = text::tidyURL($cur->cat_url,false);
		}
		
		# Check if title or url are unique
		try {
			$this->checkCategory($cur->cat_title,$cur->cat_url,$id);
		} catch (Exception $e) {
			throw $e;
		}
		
		if ($cur->cat_desc !== null) {
			$cur->cat_desc = $this->core->HTMLfilter($cur->cat_desc);
		}
	}
	
	/* ===================================================
	Entries
	=================================================== */
	/** @doc
	=== Entries management methods === */
	
	public function getPosts($params=array(),$count_only=false)
	{
		if ($count_only)
		{
			$strReq = 'SELECT count(P.post_id) ';
		}
		else
		{
			if (!empty($params['no_content'])) {
				$content_req = '';
			} else {
				$content_req =
				'post_excerpt, post_excerpt_xhtml, '.
				'post_content, post_content_xhtml, post_notes, ';
			}
			
			$strReq =
			'SELECT P.post_id, P.blog_id, P.user_id, P.cat_id, post_dt, '.
			'post_tz, post_creadt, post_upddt, post_format, post_password, '.
			'post_url, post_lang, post_title, '.$content_req.
			'post_meta, post_status, post_selected, '.
			'post_open_comment, post_open_tb, nb_comment, nb_trackback, '.
			'U.user_name, U.user_firstname, U.user_displayname, U.user_email, '.
			'U.user_url, '.
			'C.cat_title, C.cat_url ';
		}
		
		$strReq .=
		'FROM '.$this->prefix.'post P LEFT JOIN '.$this->prefix.'category C '.
		'ON P.cat_id = C.cat_id, '.
		$this->prefix.'user U ';
		
		if (!empty($params['from'])) {
			$strReq .= $params['from'].' ';
		}
		
		$strReq .=
		"WHERE P.blog_id = '".$this->con->escapeStr($this->id)."' ".
		'AND U.user_id = P.user_id ';
		
		if (!$this->core->auth->check('contentadmin',$this->id)) {
			$strReq .= 'AND ((post_status = 1 ';
			
			if ($this->without_password) {
				$strReq .= 'AND post_password IS NULL ';
			}
			$strReq .= ') ';
			
			if ($this->core->auth->userID()) {
				$strReq .= "OR P.user_id = '".$this->con->escapeStr($this->core->auth->userID())."')";
			} else {
				$strReq .= ') ';
			}
		}
		
		#Adding parameters
		if (!empty($params['post_type'])) {
			$strReq .= "AND post_type = '".$this->con->escapeStr($params['post_type'])."' ";
		} else {
			$strReq .= "AND post_type = 'post' ";
		}
		
		if (!empty($params['post_id'])) {
			$strReq .= 'AND post_id = '.(integer) $params['post_id'].' ';
		}
		
		if (!empty($params['post_url'])) {
			$strReq .= "AND post_url = '".$this->con->escapeStr($params['post_url'])."' ";
		}
		
		if (!empty($params['user_id'])) {
			$strReq .= "AND U.user_id = '".$this->con->escapeStr($params['user_id'])."' ";
		}
		
		if (!empty($params['cat_id'])) {
			$strReq .= "AND P.cat_id = ".(integer) $params['cat_id']." ";
		}
		
		if (!empty($params['cat_url'])) {
			$strReq .= "AND C.cat_url = '".$this->con->escapeStr($params['cat_url'])."' ";
		}
		
		if (isset($params['post_status'])) {
			$strReq .= 'AND post_status = '.(integer) $params['post_status'].' ';
		}
		
		if (isset($params['post_selected'])) {
			$strReq .= 'AND post_selected = '.(integer) $params['post_selected'].' ';
		}
		
		if (!empty($params['post_year'])) {
			$strReq .= 'AND '.$this->con->dateFormat('post_dt','%Y').' = '.
			"'".sprintf('%04d',$params['post_year'])."' ";
		}
		
		if (!empty($params['post_month'])) {
			$strReq .= 'AND '.$this->con->dateFormat('post_dt','%m').' = '.
			"'".sprintf('%02d',$params['post_month'])."' ";
		}
		
		if (!empty($params['post_day'])) {
			$strReq .= 'AND '.$this->con->dateFormat('post_dt','%d').' = '.
			"'".sprintf('%02d',$params['post_day'])."' ";
		}
		
		if (!empty($params['post_lang'])) {
			$strReq .= "AND P.post_lang = '".$this->con->escapeStr($params['post_lang'])."' ";
		}
		
		if (!empty($params['search']))
		{
			$words = text::splitWords($params['search']);
			if (!empty($words))
			{
				foreach ($words as $i => $w) {
					$words[$i] = "post_words LIKE '%".$this->con->escapeStr($w)."%'";
				}
				$strReq .= 'AND '.implode(' AND ',$words).' ';
			}
		}
		
		if (!empty($params['sql'])) {
			$strReq .= $params['sql'].' ';
		}
		
		if (!$count_only)
		{
			if (!empty($params['order'])) {
				$strReq .= 'ORDER BY '.$this->con->escapeStr($params['order']).' ';
			} else {
				$strReq .= 'ORDER BY post_dt DESC ';
			}
		}
		
		if (!$count_only && !empty($params['limit'])) {
			$strReq .= $this->con->limit($params['limit']);
		}
		
		try {
			$rs = $this->con->select($strReq);
			$rs->core = $this->core;
			$rs->extend('rsExtPost');
			
			# --BEHAVIOR-- coreBlogGetPosts
			$this->core->callBehavior('coreBlogGetPosts',$rs);
			
			return $rs;
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	/**
	@function getNextPost
	
	Return a recordset with post id, title and date according to the
	timestamp given in $ts. $dir could be 1 (next post) or -1 (previous post).
	
	@param	string	ts			Timestamp du billet
	@param	integer	dir			Direction de recherche
	@return	recordset
	*/
	public function getNextPost($ts,$dir)
	{
		$dt = date('YmdHis',(integer) $ts);
		
		if($dir > 0) {
               $sign = '>';
               $order = 'ASC';
          }
          else {
               $sign = '<';
               $order = 'DESC';
          }
		
		
		$params['limit'] = 1;
		$params['order'] = 'post_dt '.$order.' ';
		$params['sql'] = 
		'AND '.$this->con->dateFormat('post_dt','%Y%m%d%H%M%S').' '.$sign.' '.$dt.' ';
		
		try {
			$rs = $this->getPosts($params);
		} catch (Exception $e) {
			throw $e;
		}
		
		if ($rs->isEmpty()) {
			return null;
		}
		
		return $rs;
	}
	
	public function getLangs($params=array())
	{
		$strReq = 'SELECT COUNT(post_id) as nb_post, post_lang '.
				'FROM '.$this->prefix.'post '.
				"WHERE blog_id = '".$this->con->escapeStr($this->id)."' ".
				"AND post_lang <> '' ".
				"AND post_lang IS NOT NULL ";
		
		if (!$this->core->auth->check('contentadmin',$this->id)) {
			$strReq .= 'AND ((post_status = 1 ';
			
			if ($this->without_password) {
				$strReq .= 'AND post_password IS NULL ';
			}
			$strReq .= ') ';
			
			if ($this->core->auth->userID()) {
				$strReq .= "OR user_id = '".$this->con->escapeStr($this->core->auth->userID())."')";
			} else {
				$strReq .= ') ';
			}
		}
		
		if (isset($params['lang'])) {
			$strReq .= "AND post_lang = '".$this->con->escapeStr($params['lang'])."' ";
		}
		
		$strReq .= 'GROUP BY post_lang ';
		
		$order = 'desc';
		if (!empty($params['order']) && preg_match('/^(desc|asc)$/i',$params['order'])) {
			$order = $params['order'];
		}
		$strReq .= 'ORDER BY post_lang '.$order.' ';
		
		try {
			return $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function getDates($params=array())
	{
		$dt_f = '%Y-%m-%d';
		$dt_fc = '%Y%m%d';
		if (isset($params['type'])) {
			if ($params['type'] == 'year') {
				$dt_f = '%Y-01-01';
				$dt_fc = '%Y0101';
			} elseif ($params['type'] == 'month') {
				$dt_f = '%Y-%m-01';
				$dt_fc = '%Y%m01';
			}
		}
		$dt_f .= ' 00:00:00';
		$dt_fc .= '000000';
		
		$cat_field = $catReq = $limit = '';
		
		if (!empty($params['cat_id'])) {
			$catReq = 'AND P.cat_id = '.(integer) $params['cat_id'].' ';
			$cat_field = ', C.cat_url ';
		} elseif (!empty($params['cat_url'])) {
			$catReq = "AND C.cat_url = '".$this->con->escapeStr($params['cat_url'])."' ";
			$cat_field = ', C.cat_url ';
		}
		
		$strReq = 'SELECT DISTINCT('.$this->con->dateFormat('post_dt',$dt_f).') AS dt '.
				$cat_field.
				',COUNT(P.post_id) AS nb_post '.
				'FROM '.$this->prefix.'post P LEFT JOIN '.$this->prefix.'category C '.
				'ON P.cat_id = C.cat_id '.
				"WHERE P.blog_id = '".$this->con->escapeStr($this->id)."' ".
				$catReq;
		
		if (!$this->core->auth->check('contentadmin',$this->id)) {
			$strReq .= 'AND ((post_status = 1 ';
			
			if ($this->without_password) {
				$strReq .= 'AND post_password IS NULL ';
			}
			$strReq .= ') ';
			
			if ($this->core->auth->userID()) {
				$strReq .= "OR P.user_id = '".$this->con->escapeStr($this->core->auth->userID())."')";
			} else {
				$strReq .= ') ';
			}
		}
		
		if (!empty($params['year'])) {
			$strReq .= 'AND '.$this->con->dateFormat('post_dt','%Y')." = '".sprintf('%04d',$params['year'])."' ";
		}
		
		if (!empty($params['month'])) {
			$strReq .= 'AND '.$this->con->dateFormat('post_dt','%m')." = '".sprintf('%02d',$params['month'])."' ";
		}
		
		if (!empty($params['day'])) {
			$strReq .= 'AND '.$this->con->dateFormat('post_dt','%d')." = '".sprintf('%02d',$params['day'])."' ";
		}
		
		if (!empty($params['cat_id'])) {
			$strReq .= 'AND P.cat_id = '.(integer) $params['cat_id'].' ';
		}
		
		if (!empty($params['cat_url'])) {
			$strReq .= "AND C.cat_url = '".$this->con->escapeStr($params['cat_url'])."' ";
		}
		
		# Get next or previous date
		if (!empty($params['next']) || !empty($params['previous']))
		{
			if (!empty($params['next'])) {
				$pdir = ' > ';
				$params['order'] = 'asc';
				$dt = $params['next'];
			} else {
				$pdir = ' < ';
				$params['order'] = 'desc';
				$dt = $params['previous'];
			}
			
			$dt = date('YmdHis',strtotime($dt));
			
			$strReq .= 'AND '.$this->con->dateFormat('post_dt',$dt_fc).$pdir."'".$dt."' ";
			$limit = $this->con->limit(1);
		}
		
		$strReq .= 'GROUP BY dt '.$cat_field;
		
		$order = 'desc';
		if (!empty($params['order']) && preg_match('/^(desc|asc)$/i',$params['order'])) {
			$order = $params['order'];
		}
		
		$strReq .=
		'ORDER BY dt '.$order.' '.
		$limit;
		
		try {
			$rs = $this->con->select($strReq);
			$rs->extend('rsExtDates');
			return $rs;
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function addPost($cur)
	{
		if (!$this->core->auth->check('usage,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to create an entry'));
		}
		
		# Get ID
		try {
			$rs = $this->con->select(
				'SELECT MAX(post_id) '.
				'FROM '.$this->prefix.'post ' 
			);
		} catch (Exception $e) {
			throw $e;
		}
		
		$cur->post_id = (integer) $rs->f(0) + 1;
		$cur->blog_id = (string) $this->id;
		$cur->post_creadt = date('Y-m-d H:i:s');
		$cur->post_upddt = date('Y-m-d H:i:s');
		$cur->post_tz = $this->core->auth->getInfo('user_tz');
		
		# Post excerpt and content
		$this->getPostContent($cur,$cur->post_id);
		
		try {
			$this->getPostCursor($cur);
		} catch (Exception $e) {
			throw $e;
		}
		
		$cur->post_url = $this->getPostURL($cur->post_url,$cur->post_dt,$cur->post_title,$cur->post_id);
		
		if (!$this->core->auth->check('publish,contentadmin',$this->id)) {
			$cur->post_status = -2;
		}
		
		try {
			$cur->insert();
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
		
		return $cur->post_id;
	}
	
	public function updPost($id,&$cur)
	{
		if (!$this->core->auth->check('usage,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to update entries'));
		}
		
		$id = (integer) $id;
		
		if (empty($id)) {
			throw new Exception(__('No such entry ID'));
		}
		
		# Post excerpt and content
		$this->getPostContent($cur,$id);
		
		try {
			$this->getPostCursor($cur);
		} catch (Exception $e) {
			throw $e;
		}
		
		if ($cur->post_url !== null) {
			$cur->post_url = $this->getPostURL($cur->post_url,$cur->post_dt,$cur->post_title,$id);
		}
		
		if (!$this->core->auth->check('publish,contentadmin',$this->id)) {
			$cur->unsetField('post_status');
		}
		
		$cur->post_upddt = date('Y-m-d H:i:s');
		
		#If user is only "usage", we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT post_id '.
					'FROM '.$this->prefix.'post '.
					'WHERE post_id = '.$id.' '.
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__('You are not allowed to edit this entry'));
			}
		}
		
		try {
			$cur->update('WHERE post_id = '.$id.' ');
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function updPostStatus($id,$status)
	{
		if (!$this->core->auth->check('publish,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to change this entry status'));
		}
		
		$id = (integer) $id;
		$status = (integer) $status;
		
		#If user can only publish, we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT post_id '.
					'FROM '.$this->prefix.'post '.
					'WHERE post_id = '.$id.' '.
					"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__('You are not allowed to change this entry status'));
			}
		}
		
		$cur = $this->con->openCursor($this->prefix.'post');
		
		$cur->post_status = $status;
		$cur->post_upddt = date('Y-m-d H:i:s');
		
		try {
			$cur->update(
				'WHERE post_id = '.$id.' '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' "
			);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function updPostCategory($id,$cat_id)
	{
		if (!$this->core->auth->check('usage,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to change this entry category'));
		}
		
		$id = (integer) $id;
		$cat_id = (integer) $cat_id;
		
		#If user is only usage, we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT post_id '.
					'FROM '.$this->prefix.'post '.
					'WHERE post_id = '.$id.' '.
					"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__('You are not allowed to change this entry category'));
			}
		}
		
		$cur = $this->con->openCursor($this->prefix.'post');
		
		$cur->cat_id = ($cat_id ? $cat_id : null);
		$cur->post_upddt = date('Y-m-d H:i:s');
		
		try {
			$cur->update(
				'WHERE post_id = '.$id.' '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' "
			);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function delPost($id)
	{
		if (!$this->core->auth->check('delete,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to delete entries'));
		}
		
		$id = (integer) $id;
		
		if (empty($id)) {
			throw new Exception(__('No such entry ID'));
		}
		
		#If user can only delete, we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT post_id '.
					'FROM '.$this->prefix.'post '.
					'WHERE post_id = '.$id.' '.
					"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__('You are not allowed to delete this entry'));
			}
		}
		
		
		$strReq = 'DELETE FROM '.$this->prefix.'post '.
				'WHERE post_id = '.$id.' '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ";
		
		try {
			$this->con->execute($strReq);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function publishScheduledEntries()
	{
		$now = date('Y-m-d H:i:s');
		
		$strReq = 'SELECT post_id '.
				'FROM '.$this->prefix.'post '.
				'WHERE post_status = -1 '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
				"AND post_dt < '".$now."' ";
		
		$rs = $this->con->select($strReq);
		
		if ($rs->count() > 0)
		{
			$strReq = 'UPDATE '.$this->prefix.'post SET '.
					'post_status = 1 '.
					'WHERE post_status = -1 '.
					"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
					"AND post_dt < '".$now."' ";
			
			$this->con->execute($strReq);
			$this->triggerBlog();
		}
	}
	
	public function getPostsUsers()
	{
		$strReq = 'SELECT P.user_id, user_name, user_firstname, '.
				'user_displayname '.
				'FROM '.$this->prefix.'post P, '.$this->prefix.'user U '.
				'WHERE P.user_id = U.user_id '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
				'GROUP BY P.user_id, user_name, user_firstname, user_displayname ';
		
		try {
			$rs = $this->con->select($strReq);
		} catch (Exception $e) {
			throw $e;
		}
		
		return $rs;
	}
	
	private function getPostCursor(&$cur,$post_id=null)
	{
		if ($cur->post_title == '') {
			throw new Exception(__('No entry title'));
		}
		
		if ($cur->post_content == '') {
			throw new Exception(__('No entry content'));
		}
		
		if ($cur->post_dt == '') {
			$offset = dt::getTimeOffset($this->core->auth->getInfo('user_tz'));
			$now = time() + $offset;
			$cur->post_dt = date('Y-m-d H:i:00',$now);
		}
		
		$post_id = is_int($post_id) ? $post_id : $cur->post_id;
		
		if ($cur->post_content_xhtml == '') {
			throw new Exception(__('No entry content'));
		}
		
		# Words list
		if ($cur->post_title !== null && $cur->post_excerpt_xhtml !== null
		&& $cur->post_content_xhtml !== null)
		{
			$words =
			$cur->post_title.' '.
			$cur->post_excerpt_xhtml.' '.
			$cur->post_content_xhtml;
			
			$cur->post_words = implode(' ',text::splitWords($words));
		}
	}
	
	private function getPostContent(&$cur,$post_id)
	{
		$post_excerpt = $cur->post_excerpt;
		$post_excerpt_xhtml = $cur->post_excerpt_xhtml;
		$post_content = $cur->post_content;
		$post_content_xhtml = $cur->post_content_xhtml;
		
		$this->setPostContent(
			$post_id,$cur->post_format,$cur->post_lang,
			$post_excerpt,$post_excerpt_xhtml,
			$post_content,$post_content_xhtml
		);
		
		$cur->post_excerpt = $post_excerpt;
		$cur->post_excerpt_xhtml = $post_excerpt_xhtml;
		$cur->post_content = $post_content;
		$cur->post_content_xhtml = $post_content_xhtml;
	}
	
	public function setPostContent($post_id,$format,$lang,&$excerpt,&$excerpt_xhtml,&$content,&$content_xhtml)
	{
		if ($format == 'wiki')
		{
			$this->core->initWikiPost();
			$this->core->wiki2xhtml->setOpt('note_prefix','pnote-'.$post_id);
			if (strpos($lang,'fr') === 0) {
				$this->core->wiki2xhtml->setOpt('active_fr_syntax',1);
			}
		}
		
		if ($excerpt) {
			$excerpt_xhtml = $this->core->callFormater($format,$excerpt);
			$excerpt_xhtml = $this->core->HTMLfilter($excerpt_xhtml);
		} else {
			$excerpt_xhtml = '';
		}
		
		if ($content) {
			$content_xhtml = $this->core->callFormater($format,$content);
			$content_xhtml = $this->core->HTMLfilter($content_xhtml);
		} else {
			$content_xhtml = '';
		}
	}
	
	private function getPostURL($url,$post_dt,$post_title,$post_id)
	{
		$url = trim($url);
		
		$url_patterns = array(
		'{y}' => date('Y',strtotime($post_dt)),
		'{m}' => date('m',strtotime($post_dt)),
		'{d}' => date('d',strtotime($post_dt)),
		'{t}' => text::str2URL($post_title),
		'{id}' => (integer) $post_id
		);
		
		# If URL is empty, we create a new one
		if ($url == '')
		{
			# Transform with format
			$url = str_replace(
				array_keys($url_patterns),
				array_values($url_patterns),
				$this->settings->post_url_format
			);
		}
		else
		{
			$url = text::tidyURL($url);
		}
		
		# Let's check if URL is taken...
		$strReq = 'SELECT post_url FROM '.$this->prefix.'post '.
				"WHERE post_url = '".$this->con->escapeStr($url)."' ".
				'AND post_id <> '.(integer) $post_id. ' '.
				"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
				'ORDER BY post_url DESC';
		
		$rs = $this->con->select($strReq);
		
		if (!$rs->isEmpty())
		{
			$strReq = 'SELECT post_url FROM '.$this->prefix.'post '.
					"WHERE post_url LIKE '".$this->con->escapeStr($url)."%' ".
					'AND post_id <> '.(integer) $post_id. ' '.
					"AND blog_id = '".$this->con->escapeStr($this->id)."' ".
					'ORDER BY post_url DESC ';
			
			$rs = $this->con->select($strReq);
			
			if (preg_match('/(.*?)([0-9]+)$/',$rs->post_url,$m)) {
				$i = (integer) $m[2];
				$url = $m[1];
			} else {
				$i = 1;
			}
			
			return $url.($i+1);
		}
		
		return $url;
	}
	
	/* ===================================================
	Comments
	=================================================== */
	/** @doc
	=== Comments management methods === */
	
	public function getComments($params=array(),$count_only=false)
	{
		if ($count_only)
		{
			$strReq = 'SELECT count(comment_id) ';
		}
		else
		{
			if (!empty($params['no_content'])) {
				$content_req = '';
			} else {
				$content_req = 'comment_content, ';
			}
			
			$strReq =
			'SELECT comment_id, comment_dt, comment_tz, comment_upddt, '.
			'comment_author, comment_email, comment_site, '.
			$content_req.' comment_trackback, comment_status, '.
			'comment_spam_status, comment_ip, '.
			'P.post_title, P.post_url, P.post_id, P.post_password, '.
			'P.post_dt, P.user_id, U.user_email, U.user_url ';
		}
		
		$strReq .=
		'FROM '.$this->prefix.'comment C, '.$this->prefix.'post P, '.$this->prefix.'user U '.
		'WHERE P.post_id = C.post_id '.
		'AND U.user_id = P.user_id '.
		"AND P.blog_id = '".$this->con->escapeStr($this->id)."' ";
		
		if (!$this->core->auth->check('contentadmin',$this->id)) {
			$strReq .= 'AND ((comment_status = 1 AND P.post_status = 1 ';
			
			if ($this->without_password) {
				$strReq .= 'AND post_password IS NULL ';
			}
			$strReq .= ') ';
			
			if ($this->core->auth->userID()) {
				$strReq .= "OR P.user_id = '".$this->con->escapeStr($this->core->auth->userID())."')";
			} else {
				$strReq .= ') ';
			}
		}
		
		if (!empty($params['post_id'])) {
			$strReq .= 'AND P.post_id = '.(integer) $params['post_id'].' ';
		}
		
		if (!empty($params['cat_id'])) {
			$strReq .= 'AND P.cat_id = '.(integer) $params['cat_id'].' ';
		}
		
		if (!empty($params['comment_id'])) {
			$strReq .= 'AND comment_id = '.(integer) $params['comment_id'].' ';
		}
		
		if (isset($params['comment_status'])) {
			$strReq .= 'AND comment_status = '.(integer) $params['comment_status'].' ';
		}
		
		if (isset($params['comment_trackback'])) {
			$strReq .= 'AND comment_trackback = '.(integer) (boolean) $params['comment_trackback'].' ';
		}
		
		if (!empty($params['q_author'])) {
			$q_author = $this->con->escapeStr(str_replace('*','%',strtolower($params['q_author'])));
			$strReq .= "AND LOWER(comment_author) LIKE '".$q_author."' ";
		}
		
		if (!empty($params['sql'])) {
			$strReq .= $params['sql'].' ';
		}
		
		if (!$count_only)
		{
			if (!empty($params['order'])) {
				$strReq .= 'ORDER BY '.$this->con->escapeStr($params['order']).' ';
			} else {
				$strReq .= 'ORDER BY comment_dt DESC ';
			}
		}
		
		if (!$count_only && !empty($params['limit'])) {
			$strReq .= $this->con->limit($params['limit']);
		}
		
		try {
			$rs = $this->con->select($strReq);
			$rs->core = $this->core;
			$rs->extend('rsExtComment');
			
			# --BEHAVIOR-- coreBlogGetComments
			$this->core->callBehavior('coreBlogGetComments',$rs);
			
			return $rs;
		} catch (Exception $e) {
			throw $e;
		}
		
		
	}
	
	public function addComment($cur)
	{
		# Get ID
		try {
			$rs = $this->con->select(
				'SELECT MAX(comment_id) '.
				'FROM '.$this->prefix.'comment ' 
			);
		} catch (Exception $e) {
			throw $e;
		}
		
		$cur->comment_id = (integer) $rs->f(0) + 1;
		$cur->comment_upddt = date('Y-m-d H:i:s');
		
		$offset = dt::getTimeOffset($this->settings->blog_timezone);
		$cur->comment_dt = date('Y-m-d H:i:s',time() + $offset);
		$cur->comment_tz = $this->settings->blog_timezone;
		
		try {
			$this->getCommentCursor($cur);
		} catch (Exception $e) {
			throw $e;
		}
		
		if ($cur->comment_ip === null) {
			$cur->comment_ip = http::realIP();
		}
		
		try {
			$cur->insert();
			$this->triggerComment($cur->comment_id);
			$this->triggerBlog();
			return $cur->comment_id;
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function updComment($id,&$cur)
	{
		if (!$this->core->auth->check('usage,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to update comments'));
		}
		
		$id = (integer) $id;
		
		if (empty($id)) {
			throw new Exception(__('No such comment ID'));
		}
		
		#If user is only usage, we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT P.post_id '.
					'FROM '.$this->prefix.'post P, '.$this->prefix.'comment C '.
					'WHERE P.post_id = C.post_id '.
					"AND P.blog_id = '".$this->con->escapeStr($this->id)."' ".
					'AND comment_id = '.$id.' '.
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__('You are not allowed to update this comment'));
			}
		}
		
		try {
			$this->getCommentCursor($cur);
		} catch (Exception $e) {
			throw $e;
		}
		
		$cur->comment_upddt = date('Y-m-d H:i:s');
		
		if (!$this->core->auth->check('publish,contentadmin',$this->id)) {
			$cur->unsetField('comment_status');
		}
		
		try {
			$cur->update('WHERE comment_id = '.$id.' ');
			$this->triggerComment($id);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function updCommentStatus($id,$status)
	{
		if (!$this->core->auth->check('publish,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to change this entry status'));
		}
		
		$id = (integer) $id;
		$status = (integer) $status;
		
		# If user can only publish, we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT P.post_id '.
					'FROM '.$this->prefix.'post P, '.$this->prefix.'comment C '.
					'WHERE P.post_id = C.post_id '.
					"AND P.blog_id = '".$this->con->escapeStr($this->id)."' ".
					'AND comment_id = '.$id.' '.
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__("You are not allowed to change this comment's status"));
			}
		}
		
		$cur = $this->con->openCursor($this->prefix.'comment');
		
		$cur->comment_status = $status;
		$cur->comment_upddt = date('Y-m-d H:i:s');
		
		try {
			$cur->update('WHERE comment_id = '.$id);
			$this->triggerComment($id);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	public function delComment($id)
	{
		if (!$this->core->auth->check('delete,contentadmin',$this->id)) {
			throw new Exception(__('You are not allowed to delete comments'));
		}
		
		$id = (integer) $id;
		
		if (empty($id)) {
			throw new Exception(__('No such comment ID'));
		}
		
		#If user can only delete, we need to check the post's owner
		if (!$this->core->auth->check('contentadmin',$this->id))
		{
			$strReq = 'SELECT P.post_id '.
					'FROM '.$this->prefix.'post P, '.$this->prefix.'comment C '.
					'WHERE P.post_id = C.post_id '.
					"AND P.blog_id = '".$this->con->escapeStr($this->id)."' ".
					'AND comment_id = '.$id.' '.
					"AND user_id = '".$this->con->escapeStr($this->core->auth->userID())."' ";
			try {
				$rs = $this->con->select($strReq);
			} catch (Exception $e) {
				throw $e;
			}
			
			if ($rs->isEmpty()) {
				throw new Exception(__('You are not allowed to delete this comment'));
			}
		}
		
		$strReq = 'DELETE FROM '.$this->prefix.'comment '.
				'WHERE comment_id = '.$id.' ';
		
		try {
			$this->triggerComment($id,true);
			$this->con->execute($strReq);
			$this->triggerBlog();
		} catch (Exception $e) {
			throw $e;
		}
	}
	
	private function getCommentCursor(&$cur)
	{
		if ($cur->comment_content !== null && $cur->comment_content == '') {
			throw new Exception(__('You must provide a comment'));
		}
		
		if ($cur->comment_author !== null && $cur->comment_author == '') {
			throw new Exception(__('You must provide an author name'));
		}
		
		if ($cur->comment_email != '' && !text::isEmail($cur->comment_email)) {
			throw new Exception(__('Email adress is not valid.'));
		}
		
		if ($cur->comment_site !== null && $cur->comment_site != '') {
			if (!preg_match('|^http(s?)://|',$cur->comment_site)) {
				$cur->comment_site = 'http://'.$cur->comment_site;
			}
		}
		
		if ($cur->comment_status === null) {
			$cur->comment_status = (integer) $this->settings->comments_pub;
		}
		
		# Words list
		if ($cur->comment_content !== null)
		{
			$cur->comment_words = implode(' ',text::splitWords($cur->comment_content));
		}
	}
}
?>