<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of Clearbricks.
# Copyright (c) 2004 Olivier Meunier and contributors. All rights
# reserved.
#
# Clearbricks is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# Clearbricks is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Clearbricks; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****

require dirname(__FILE__).'/class.cursor.php';

interface i_dbLayer
{
	function db_connect($host,$user,$password,$database);
	
	function db_pconnect($host,$user,$password,$database);
	
	function db_close($handle);
	
	function db_version($handle);
	
	function db_query($handle,$query);
	
	function db_exec($handle,$query);
	
	function db_num_fields($res);
	
	function db_num_rows($res);
	
	function db_field_name($res,$position);
	
	function db_field_type($res,$position);
	
	function db_fetch_assoc($res);
	
	function db_result_seek($res,$row);
	
	function db_changes($handle,$res);
	
	function db_get_tables($handle);
	
	function db_get_columns($handle,$table);
	
	function db_last_error($handle);
	
	function db_escape_string($str,$handle=null);
}

class dbLayer
{
	protected $__driver = null;
	protected $__version = null;
	
	protected $__link;
	protected $__last_result;
	
	/**
	@function __construct
	
	Inits database connection.
	
	@param	string	host			User ID
	@param	string	database		Password
	@param	string	user			Server to connect
	@param	string	password		Database name
	*/
	public function __construct($host,$database,$user='',$password='',$persistent=false)
	{
		if ($persistent) {
			$this->__link = $this->db_pconnect($host,$user,$password,$database);
		} else {
			$this->__link = $this->db_connect($host,$user,$password,$database);
		}
		
		$this->__version = $this->db_version($this->__link);
	}
	
	/**
	@function close
	
	Closes database connection.
	*/
	public function close()
	{
		$this->db_close($this->__link);
	}
	
	/**
	@function driver
	
	Returns database driver name
	
	@return	string
	*/
	public function driver()
	{
		return $this->__driver;
	}
	
	/**
	@function version
	
	Returns database driver version
	
	@return	string
	*/
	public function version()
	{
		return $this->__version;
	}
	
	/**
	@function link
	
	Returns link resource
	
	@return	resource
	*/
	public function link()
	{
		return $this->__link;
	}
	
	/**
	@function select
	
	Executes a query and return a recordset. Recordset could be either static
	(default beahvior) or dynamic (useful for large results).
	$query could be a string or an array of params for a previously prepared
	query.
	
	@param	string		query		SQL query
	@return	recordset
	*/
	public function select($sql)
	{
		$result = $this->db_query($this->__link,$sql);
		
		$this->__last_result =& $result;
		
		$info = array();
		$info['con'] =& $this;
		$info['cols'] = $this->db_num_fields($result);
		$info['rows'] = $this->db_num_rows($result);
		$info['info'] = array();
		
		for ($i=0; $i<$info['cols']; $i++) {
			$info['info']['name'][] = $this->db_field_name($result,$i);
			$info['info']['type'][] = $this->db_field_type($result,$i);
		}
		
		return new record($result,$info);
	}
	
	/**
	@function execute
	
	Executes a query and return true if query succeded.
	
	@param	string		query		SQL query
	@return	boolean
	*/
	public function execute($sql)
	{
		$result = $this->db_exec($this->__link,$sql);
		
		$this->__last_result =& $result;
		
		return true;
	}
	
	/**
	@function begin
	
	Begins a transaction.
	*/
	public function begin()
	{
		$this->execute('BEGIN');
	}
	
	/**
	@function commit
	
	Commits a transaction.
	*/
	public function commit()
	{
		$this->execute('COMMIT');
	}
	
	/**
	@function rollback
	
	Rollbacks a transaction.
	*/
	public function rollback()
	{
		$this->execute('ROLLBACK');
	}
	
	/**
	@function vacuum
	
	Vacuum the table given in argument.
	
	@param	string	table		Table name
	*/
	public function vacuum($table)
	{
	}
	
	/**
	@function getTables
	
	Returns an array of all table names.
	
	@return	array
	*/
	public function getTables()
	{
		return $this->db_get_tables($this->__link);
	}
	
	/**
	@function getColumns
	
	Return an array of columns (name and type) of a given table.
	
	@param	string	table		Table name
	@return	array
	*/
	public function getColumns($table)
	{
		return $this->db_get_columns($this->__link,$table);
	}
	
	/**
	@function changes
	
	Returns number of affected lines by the last DELETE, INSERT or UPDATE
	query.
	
	@return	integer
	*/
	public function changes()
	{
		return $this->db_changes($this->__link,$this->__last_result);
	}
	
	/**
	@function error
	
	Returns last database error or false if no error.
	
	@return string
	*/
	public function error()
	{
		$err = $this->db_last_error($this->__link);
		
		if (!$err) {
			return false;
		}
		
		return $err;
	}

	/**
	@function dateFormat
	
	Returns a query fragment with date formater.
	
	Following modifiers are accepted:
	
	%d : Day of the month, numeric
	%H : Hour 24 (00..23)
	%M : Minute (00..59)
	%m : Month numeric (01..12)
	%S : Seconds (00..59)
	%Y : Year, numeric, four digits
	
	@param	string	field		Field name
	@param	string	patern		Date format
	@return	string
	*/
	public function dateFormat($field,$pattern)
	{
		return
		'TO_CHAR('.$field.','."'".$this->escapeStr($pattern)."') ";
	}
	
	/**
	@function limit
	
	Returns a LIMIT query fragment.
	
	@param	mixed	arg1		array or integer with limit intervals
	@param	mixed	arg2		integer or null (null)
	@return	string
	*/
	public function limit($arg1, $arg2=null)
	{
		if (is_array($arg1))
		{
			$arg1 = array_values($arg1);
			$arg2 = isset($arg1[1]) ? $arg1[1] : null;
			$arg1 = $arg1[0];
		}
		
		if ($arg2 === null) {
			$sql = ' LIMIT '.(integer) $arg1.' ';
		} else {
			$sql = ' LIMIT '.(integer) $arg2.' OFFSET '.$arg1.' ';
		}
		
		return $sql;
	}
	
	/**
	@function escapeStr
	
	Returns SQL protected string.
	
	@param	string	str			String to protect
	@return	string
	*/
	public function escapeStr($str)
	{
		return $this->db_escape_string($str,$this->__link);
	}
	
	/**
	@function escapeSystem
	
	Returns SQL system protected string.
	
	@param	string	str			String to protect
	@return	string
	*/
	public function escapeSystem($str)
	{
		return '"'.$str.'"';
	}
	
	public function openCursor($table)
	{
		return new cursor($this,$table);
	}
}

class record
{
	protected $__link;
	protected $__result;
	protected $__info;
	protected $__extend = array();
	
	protected $__index = 0;
	private $__fetch = false;
	protected $__row = false;
	
	
	public function __construct($result,$info)
	{
		$this->__result = $result;
		$this->__info = $info;
		$this->__link = $info['con']->link();
		$this->index(0);
	}
	
	public function toStatic()
	{
		return new staticRecord($this->__result,$this->__info);
	}
	
	public function __call($f,$args)
	{
		if (isset($this->__extend[$f]))
		{
			array_unshift($args,$this);
			return call_user_func_array($this->__extend[$f],$args);
		}
		
		trigger_error('Call to undefined method record::'.$f.'()',E_USER_ERROR);
	}
	
	public function __get($n)
	{
		return $this->field($n);
	}
	
	public function f($n)
	{
		return $this->field($n);
	}
	
	public function field($n)
	{
		return $this->__row[$n];
	}
	
	public function exists($field)
	{
		return isset($this->__row[$field]);
	}
	
	public function extend($class)
	{
		if (!class_exists($class)) {
			return;
		}
		
		$c = new ReflectionClass($class);
		foreach ($c->getMethods() as $m) {
			if ($m->isStatic() && $m->isPublic()) {
				$this->__extend[$m->name] = array($class,$m->name);
			}
		}
	}
	
	private function setRow()
	{
		$this->__row = $this->__info['con']->db_fetch_assoc($this->__result);
		
		if ($this->__row !== false)
		{
			foreach ($this->__row as $k => $v) {
				$this->__row[] =& $this->__row[$k];
			}
			return true;
		}
		else
		{
			return false;
		}
	}
	
	public function index($row=null)
	{
		if ($row === null) {
			return $this->__index === null ? 0 : $this->__index;
		}
		
		if ($row < 0 || $row+1 > $this->__info['rows']) {
			return false;
		}
		
		if ($this->__info['con']->db_result_seek($this->__result,(integer) $row))
		{
			$this->__index = $row;
			$this->setRow();
			$this->__info['con']->db_result_seek($this->__result,(integer) $row);
			return true;
		}
		return false;
	}
	
	public function fetch()
	{
		if (!$this->__fetch) {
			$this->__fetch = true;
			$i = -1;
		} else {
			$i = $this->__index;
		}
		
		if (!$this->index($i+1)) {
			$this->__fetch = false;
			$this->__index = 0;
			return false;
		}
		
		return true;
	}
	
	public function moveStart()
	{
		return $this->index(0);
	}
	
	public function moveEnd()
	{
		return $this->index($this->__info['rows']-1);
	}
	
	public function moveNext()
	{
		return $this->index($this->__index+1);
	}
	
	public function movePrev()
	{
		return $this->index($this->__index-1);
	}
	
	public function isEnd()
	{
		return $this->__index+1 == $this->count();
	}
	
	public function isStart()
	{
		return $this->__index <= 0;
	}
	
	public function isEmpty()
	{
		return $this->count() == 0;
	}
	
	public function count()
	{
		return $this->__info['rows'];
	}
	
	public function columns()
	{
		return $this->__info['info']['name'];
	}
	
	public function rows()
	{
		return $this->getData();
	}
	
	protected function getData()
	{
		$res = array();
		
		if ($this->count() == 0) {
			return $res;
		}
		
		$this->__info['con']->db_result_seek($this->__result,0);
		while (($r = $this->__info['con']->db_fetch_assoc($this->__result)) !== false) {
			foreach ($r as $k => $v) {
				$r[] =& $r[$k];
			}
			$res[] = $r;
		}
		$this->__info['con']->db_result_seek($this->__result,$this->__index);
		
		return $res;
	}
}

class staticRecord extends record
{
	public $__data = array();
	private $__sortfield;
	private $__sortsign;
	
	public function __construct($result,$info)
	{
		if (is_array($result))
		{
			$this->__info = $info;
			$this->__data = $result;
		}
		else
		{
			parent::__construct($result,$info);
			$this->__data = parent::getData();
		}
		
		unset($this->__link);
		unset($this->__result);
	}
	
	public static function newFromArray($data)
	{
		if (!is_array($data)) {
			$data = array();
		}
		
		$data = array_values($data);
		
		if (empty($data) || !is_array($data[0])) {
			$cols = 0;
		} else {
			$cols = count($data[0]);
		}
		
		$info = array(
			'con' => null,
			'info' => null,
			'cols' => $cols,
			'rows' => count($data)
		);
		
		return new self($data,$info);
	}
	
	public function field($n)
	{
		return $this->__data[$this->__index][$n];
	}
	
	public function index($row=null)
	{
		if ($row === null) {
			return $this->__index;
		}
		
		if ($row < 0 || $row+1 > $this->__info['rows']) {
			return false;
		}
		
		$this->__index = $row;
		return true;
	}
	
	public function rows()
	{
		return $this->__data;
	}
	
	public function set($n,$v)
	{
		if ($this->__index === null) {
			return false;
		}
		
		$this->__data[$this->__index][$n] = $v;
	}
	
	public function sort($field,$order='asc')
	{
		if (!isset($this->__data[0][$field])) {
			return false;
		}
		
		$this->__sortfield = $field;
		$this->__sortsign = strtolower($order) == 'asc' ? 1 : -1;
		
		usort($this->__data,array($this,'sortCallback'));
		
		$this->__sortfield = null;
		$this->__sortorder = null;
	}
	
	private function sortCallback($a,$b)
	{
		return strcmp($a[$this->__sortfield],$b[$this->__sortfield]) * $this->__sortsign;
	}
}
?>