<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of Clearbricks.
# Copyright (c) 2004 Florent Cotton, Olivier Meunier and contributors.
# All rights reserved.
#
# Clearbricks is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# Clearbricks is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Clearbricks; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****

require_once dirname(__FILE__).'/../ext/incutio.http_client.php';
require_once dirname(__FILE__).'/class.feed.parser.php';

class feedReader extends HttpClient
{
	protected $user_agent = 'Clearbricks Feed Reader/0.2';
	protected $validators = array();
	
	private $cache_dir = null;
	private $cache_file_prefix = 'cbfeed_';
	private $cache_ttl = '-30 minutes';
	
	public function __construct()
	{
		parent::__construct('');
	}
	
	public function parse($url)
	{
		if ($this->cache_dir)
		{
			return $this->withCache($url);
		}
		else
		{
			if (!$this->getFeed($url)) {
				return false;
			}
			
			if ($this->getStatus() != '200') {
				return false;
			}
			
			return new feedParser($this->getContent());
		}
	}
	
	public static function quickParse($url,$cache_dir=null)
	{
		$parser = new self();
		if ($cache_dir) {
			$parser->setCacheDir($cache_dir);
		}
		
		return $parser->parse($url);
	}
	
	public function setCacheDir($dir)
	{
		$this->cache_dir = null;
		
		if (!empty($dir) && is_dir($dir) && is_writeable($dir))
		{
			$this->cache_dir = $dir;
			return true;
		}
		
		return false;
	}
	
	public function setCacheTTL($str)
	{
		$str = trim($str);
		if (!empty($str))
		{
			if (substr($str,0,1) != '-') {
				$str = '-'.$str;
			}
			$this->cache_ttl = $str;
		}
	}
	
	protected function getFeed($url)
	{
		$bits = parse_url($url);
		$this->host = $bits['host'];
		$this->port = isset($bits['port']) ? $bits['port'] : 80;
		$path = isset($bits['path']) ? $bits['path'] : '/';
		
		if (isset($bits['query'])) {
			$path .= '?'.$bits['query'];
		}
		
		if (!$this->get($path)) {
			return false;
		}
		
		return true;
	}
	
	protected function withCache($url)
	{
		$cached_file = $this->cache_dir.'/'.$this->cache_file_prefix.md5($url);
		$may_use_cached = false;
		
		if (@file_exists($cached_file))
		{
			$may_use_cached = true;
			$ts = @filemtime($cached_file);
			if ($ts > strtotime($this->cache_ttl))
			{
				// Direct cache
				return unserialize(file_get_contents($cached_file));
			}
			$this->setValidator('IfModifiedSince', $ts);
		}
		
		if (!$this->getFeed($url))
		{
			if ($may_use_cached)
			{
				// connection failed - fetched from cache
				return unserialize(file_get_contents($cached_file));
			}
			return false;
		}
		
		switch ($this->getStatus())
		{
			case '304':
				@touch($cached_file);
				return unserialize(file_get_contents($cached_file));
			case '200':
				if ($feed = new feedParser($this->getContent()))
				{
					if (($fp = @fopen($cached_file, 'w')))
					{
						fwrite($fp, serialize($feed));
						fclose($fp);
					}
					return $feed;
				}
		}
		
		return false;
	}
	
	# Cache control HttpClient extension
	protected function buildRequest()
	{
		$r = parent::buildRequest();
		$r = preg_replace("%\r\n\r\n\$%",'',$r);
		
		// Cache validators
		$headers = array();
		if (!empty($this->validators))
		{
			if (isset($this->validators['IfModifiedSince'])) {
				$headers[] = 'If-Modified-Since: '.$this->validators['IfModifiedSince'];
			}
			if (isset($this->validators['IfNoneMatch'])) {
				if (is_array($this->validators['IfNoneMatch'])) {
					$etags = implode(',',$this->validators['IfNoneMatch']);
				} else {
					$etags = $this->validators['IfNoneMatch'];
				}
				$headers[] = '';
			}
		}
		
		return $r."\r\n".implode("\r\n", $headers)."\r\n\r\n";
	}
	
	private function setValidator($key,$value)
	{
		if ($key == 'IfModifiedSince') {
			$value = gmdate('D, d M Y H:i:s',$value).' GMT';
		}
		
		$this->validators[$key] = $value;
	}
}
?>